


/*******************************************************************
/* DP program v3.1                                                 */
/* Created by Ivan Zorych, e-mail: zorych@gmail.com    	           */
/*                       http://www.data-and-statistics.com        */
*******************************************************************/

/*************************************************************************/
/*(c) 2009-2012 Foundation for the National Institutes of Health (FNIH). */
/* Licensed under the Apache License, Version 2.0 (the "License");       */
/* you may not use this file except in compliance with the License.      */
/* You may obtain a copy of the License at                               */
/* http://omop.fnih.org/publiclicense.                                   */
/* Unless required by applicable law or agreed to in writing,            */
/* software distributed under the License is distributed on an           */
/* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,          */
/* either express or implied. Any redistributions of this work or        */
/* any derivative work or modification based on this work should         */
/* be accompanied by the following source attribution:                   */
/* "This work is based on work by the Observational Medical Outcomes     */
/* Partnership (OMOP) and used under license from the FNIH at            */
/* http://omop.fnih.org/publiclicense.                                   */
/* Any scientific publication that is based on this work should          */
/* include a reference to                                                */
/* http://omop.fnih.org.                                                 */
/*************************************************************************/;

options notes;

/*******************************************************************
* DEFINE LIBRARIES and DIRECTORIES                                 *
*******************************************************************/


/*  Define the name (abbreviation) of the database, up to 4 letters; (edit only right side) */
%let dbn_ = OSIM;

/* Point to Data in the CDM format (edit only right side) */
%let cdmdata = F:\data1\OSIM11_50K;

/* Point to the work directory (edit only right side) */
%let wrkfolder = F:\DP_WRK;

/* Point to directory that contains parameter file, cc_parameters.txt, 
and (optionally) drugs_of_interest.txt, conditions_of_interest.txt. 
If either of optional files exists, it will influence the output and performance time  
(edit only right side)      */
%let Parameters_here =E:\0001_METHODS\DP\DP_METHOD\PARAMS;
 

/* Point to the output directory (edit only right side)*/
%let outfolder =F:\DP_WRK;

/* Point to the directory for the output comma delimited file (edit only right side */
%let TEXT_OUT_DIR = F:\DP_WRK;


/* DO NOT EDIT BELOW THIS LINE */
options THREADS CPUCOUNT=Actual;

LIBNAME SOURCE "&cdmdata";
LIBNAME WRK "&wrkfolder";
LIBNAME OUT "&outfolder";

%let DATAIN=SOURCE;   /* LIBRARY SHOULD CONTAIN DRUG_ERA, CONDITON_ERA, AND PERSON FILES/TABLES. */

%let DATAWRK=WRK;  /* DEFINE THE NAME OF THE WORKING LIBRARY */

%let DATAOUT=OUT;   /* DEFINE THE NAME OF THE OUTPUT LIBRARY */


%let DP_parameters=&Parameters_here./DP_parameters.txt;
%let drugs_of_interest=&Parameters_here./drugs_of_interest.txt;
%let conditions_of_interest=&Parameters_here./conditions_of_interest.txt;


options CPUCOUNT=ACTUAL THREADS;


%macro c_s1(datawrk, dataout, cond_type, p_window_drugs, p_window_cond, sex_s, age_s, year_s, dbn, pvn);
/*******************************************************************
* MACRO c_s1 calculates 1st counting scenario ("distinct patients")*
* INPUT PARAMETERS:                                                *
* datawrk - library that stores working files and preprocessed     *
* data in the cdm format                                           *
* dataout - library for the output file(s)                         *
* cond_type - condition type: 1= prevalent, 2=incident             *
* p_window_drugs - drug persistence window: 0 or 30                *
* p_window_cond - condition persistence window: 0 or 30            *
* sex_s - stratify by sex: 0 or 1                                  *
* age_s - stratify by age: 0 or 1                                  *
* year_s - stratify by year: 0 or 1                                *
* dbn - four letter abbreviation of the database name              *
*******************************************************************/



/*******************************************************************
*          Unique drugs, conditions by Person                       *
*******************************************************************/
proc sort data=&DATAWRK..Drug_era (keep=Person_id Drug_concept_id) out=&DATAWRK..Drug_era_uni NODUPKEY THREADS;
by Person_id Drug_concept_id;
run;
proc sort data=&DATAWRK..Condition_era ( keep=Person_id Condition_concept_id) out=&DATAWRK..Condition_era_uni NODUPKEY THREADS;
by Person_id Condition_concept_id;
run;

%if &cond_type=1 %then
         %do; %let Condition_File=Condition_era;  %end;
%if &cond_type=2 %then
%do; %let Condition_File=Condition_era_min; 

   
                proc sql THREADS;
                create table &DATAWRK..Condition_era_min as
                select T1.Person_id, T1.Condition_concept_id, min(T1.Condition_era_start_date)
                as Condition_era_start_date format DATE9.
                from  &DATAWRK..Condition_era T1
                group by T1.Person_id, T1.Condition_concept_id;
                quit;
        
%end;
/*******************************************************************
*          Stratification                                          *
*******************************************************************/
%let st=%eval(&sex_s+&age_s+&year_s);

%if &st ^= 0 %then
%do;

        proc sort data=&DATAWRK..&Condition_File;
        by Person_id;
        run;

        proc sort data=&DATAWRK..Person;
        by Person_id; run;

        data &DATAWRK..S1;
        merge &DATAWRK..First_record (IN=SET1) &DATAWRK..Person (IN=SET2);
        by Person_id;
        
        %if &age_s =1 %then
        %do;
        if (year_of_birth NE .) AND (year NE .) then age = year - year_of_birth;
        else age = -999;
                %end;

                if SET1 AND SET2;
        %if &sex_s = 0 %then %do; drop  Gender_concept_id; %end;
        %if &year_s = 0 %then %do; drop  year; %end;
        drop Year_of_birth; run;

                        %if &age_s = 1 %then
            %do;

                        data &DATAWRK..all_positive_years;
                        set &DATAWRK..S1;
                        where age >=0;
                        keep age;
                        run;


                        proc univariate data=&DATAWRK..all_positive_years noprint;
                        var age;
                        output out=&DATAWRK..pntls pctlpre=P_ pctlpts= 12.5 to 87.5 by 12.5;
                        run;


                        data _null_;
                        set &DATAWRK..pntls;
                        call symput('PNTL_12_5',p_12_5);
                        call symput('PNTL_25',p_25);
                        call symput('PNTL_37_5',p_37_5);
                        call symput('PNTL_50',p_50);
                        call symput('PNTL_62_5',p_62_5);
                        call symput('PNTL_75',p_75);
                        call symput('PNTL_87_5',p_87_5);
                run;

                        %put &PNTL_12_5 &PNTL_25 &PNTL_37_5 &PNTL_50 &PNTL_62_5 &PNTL_75 &PNTL_87_5;
 
                
                data &DATAWRK..S1;
                        set &DATAWRK..S1;
                        if age < 0 then age_group=0;
                        else if 0<= age <= &PNTL_12_5 then age_group=1;
                else if &PNTL_12_5< age <= &PNTL_25 then age_group=2;   
                else if &PNTL_25< age <= &PNTL_37_5 then age_group=3;
                        else if &PNTL_37_5< age <= &PNTL_50 then age_group=4;
                else if &PNTL_50< age <= &PNTL_62_5 then age_group=5;
                        else if &PNTL_62_5< age <= &PNTL_75 then age_group=6;
                else if &PNTL_75< age <= &PNTL_87_5 then age_group=7;
                        else if &PNTL_87_5< age then age_group=8;
                run;
                        %end;

        proc sort data=&DATAWRK..S1
        (keep=
        %if &sex_s = 1  %then  Gender_concept_id;
        %if &age_s = 1  %then  age_group;
        %if &year_s = 1 %then  year;)
        out=&DATAWRK..Strata NODUPKEY THREADS;
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ; run;

        data &DATAWRK..Strata; set &DATAWRK..Strata;
        stratum=_n_; run;

        proc sort data=&DATAWRK..S1;
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ; run;

        data &DATAWRK..S1;
        merge &DATAWRK..S1 &DATAWRK..Strata;
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ;
        keep Person_id Stratum;
        run;
%end;

%else
%do;
        %let str=0;
        data &DATAWRK..Strata;  stratum=1; run;

        data &DATAWRK..S1;      set  &DATAWRK..Person;
        stratum=1;      keep Person_id stratum; run;
%end;

proc sort data=&DATAWRK..S1 THREADS; by Person_id; run;
proc sort data=&DATAWRK..&Condition_File THREADS; by Person_id; run;

data &DATAWRK..Condition_current;
merge &DATAWRK..&Condition_File (IN=SET1) &DATAWRK..S1 (IN=SET2);
by Person_id;
if SET1 AND SET2;
run;

data &DATAWRK..Drug_current;
merge &DATAWRK..Drug_era (IN=SET1) &DATAWRK..S1 (IN=SET2);
by Person_id;
if SET1 AND SET2;
run;

data &DATAWRK..Drug_era_uni;
merge &DATAWRK..Drug_era_uni (IN=SET1) &DATAWRK..S1 (IN=SET2);
by Person_id;
if SET1 AND SET2;
run;

data &DATAWRK..Condition_era_uni;
merge &DATAWRK..Condition_era_uni (IN=SET1) &DATAWRK..S1 (IN=SET2);
by Person_id;
if SET1 AND SET2;
run;

/*******************************************************************
*          END OF STRATIFICATION                                   *
*******************************************************************/


/*******************************************************************
*          drug x condition counts: W00                            *
*******************************************************************/

proc sql THREADS;
create table &DATAWRK..DPTable_1 as
select T1.Drug_concept_id, T2.Condition_concept_id, T1.stratum,  count(T1.Person_id) as W00
from (select DISTINCT T1.Person_id, T1.Drug_concept_id, T2.Condition_concept_id
from &DATAWRK..Drug_current T1, &DATAWRK..Condition_current T2
where T1.Person_id=T2.Person_id AND
T1.Drug_era_start_date<= T2.Condition_era_start_date <= T1.Drug_era_end_date)
group by T1.Drug_concept_id, T2.Condition_concept_id, T1.stratum;
quit;



/*******************************************************************
*          AB=W00+W11, 'take drug'                                 *
*          use to find  W11                                        *
*******************************************************************/
proc sql THREADS;
create table &DATAWRK..DPTable_2s1 as
select T1.Drug_concept_id, T1.stratum, count(T1.Person_id) as AB
from &DATAWRK..Drug_era_uni T1
group by T1.Drug_concept_id, T1.stratum;
quit;


/*******************************************************************
*          calculate W01 as AB - W00                               *
*******************************************************************/
proc sql THREADS;
create table &DATAWRK..DPtable_3 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, T2.AB, T2.AB-T1.W00 as W01
from &DATAWRK..DPtable_1 T1, &DATAWRK..DPtable_2s1 T2
where (T1.Drug_concept_id=T2.Drug_concept_id) AND (T1.stratum=T2.stratum);
quit;


/*******************************************************************
* C1= 'number of distinct patients that have a condition'          *
* used to calculate W10;  calculate W01 as AB - W00                *
*******************************************************************/

        proc sql THREADS;
        create table &DATAWRK..DPTable_3s1 as
        select T1.Condition_concept_id, T1.stratum, count(T1.Person_id) as C1
        from &DATAWRK..Condition_era_uni T1
        group by T1.Condition_concept_id, T1.stratum;
        quit;

proc sql THREADS;
create table &DATAWRK..DPtable_4 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, T1.W01, T1.AB, T2.C1
from &DATAWRK..DPtable_3 T1, &DATAWRK..DPtable_3s1 T2
where (T1.Condition_concept_id=T2.Condition_concept_id) AND (T1.stratum = T2.stratum);
quit;


/*******************************************************************
* calculate C2 ='number of distinct patients that had              *
* a drug and a condition'                                          *
*******************************************************************/
        proc sql THREADS;  /*for W10,  C2 */
        create table &DATAWRK..DPTable_4s1 as
        select T1.Drug_concept_id, T2.Condition_concept_id, T1.stratum, count (T1.Person_id) as C2
        from    &DATAWRK..Drug_era_uni T1, &DATAWRK..Condition_era_uni T2
        where T1.Person_id=T2.Person_id
        group by T1.Drug_concept_id, T2.Condition_concept_id, T1.stratum;
        quit;

/*******************************************************************
* calculate W10, store W00, W01, and W10                           *
*******************************************************************/

proc sql THREADS;
create table &DATAWRK..DPtable_5 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, T1.W01, T1.C1-T2.C2 as W10
from &DATAWRK..DPtable_4 T1, &DATAWRK..DPtable_4s1 T2
where T1.Drug_concept_id=T2.Drug_concept_id AND T1.Condition_concept_id=T2.Condition_concept_id
AND T1.stratum=T2.stratum;
quit;


/*******************************************************************
* just N, total number of patients by stratum                      *
*******************************************************************/
        proc sql THREADS;
        create table &DATAWRK..DPTable_5s1 as
        select T1.stratum, count(T1.Person_id) as N
        from &DATAWRK..S1 T1
        group by T1.stratum;
        quit;

/*******************************************************************
* all counts: W00, W01, W10, W11                                   *
*******************************************************************/


proc sql THREADS;
create table &DATAOUT..DP&pvn.&cond_type.1d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, T1.W01, T1.W10, T2.N - T1.W00 -T1.W01 -T1.W10 AS W11,T2.N
from &DATAWRK..DPtable_5 T1, &DATAWRK..DPtable_5s1 T2
where T1.stratum = T2.stratum;
quit;
%mend c_s1;


%macro c_s2(datawrk, dataout, cond_type, p_window_drugs, p_window_cond, sex_s, age_s, year_s, dbn, pvn);
/*******************************************************************
* MACRO c_s2 calculates 2nd counting scenario ("SRS like")         *
* for both condition types, prevalent and incident;                *
* INPUT PARAMETERS:                                                *
* datawrk - library that stores working files and preprocessed     *
* data in the cdm format                                           *
* dataout - library for the output file(s)                         *
* cond_type - condition type: 1= prevalent, 2=incident             *
* p_window_drugs - drug persistence window: 0 or 30                *
* p_window_cond - condition persistence window: 0 or 30            *
* sex_s - stratify by sex: 0 or 1                                  *
* age_s - stratify by age: 0 or 1                                  *
* year_s - stratify by year: 0 or 1                                *
* dbn - four letter abbreviation of the database name              *
*******************************************************************/


/*******************************************************************
*          Prepare data for Prevalent/Incident conditions          *
*******************************************************************/
%if &cond_type=1 %then
%do; %let Condition_File=Condition_era; %let Condition_File_S=Condition_era_P;
         %let Strata=Strata_P;
%end;

%if &cond_type=2 %then
%do; %let Condition_File=Condition_era_min; %let Condition_File_S=Condition_era_I;
        %let Strata=Strata_I;
        
        proc sql THREADS;
        create table &DATAWRK..Condition_era_min as
        select T1.Person_id, T1.Condition_concept_id, min(T1.Condition_era_start_date)
        as Condition_era_start_date format DATE9.
        from  &DATAWRK..Condition_era T1
        group by T1.Person_id, T1.Condition_concept_id;
        quit;
        
%end;
/*******************************************************************
*          Stratification                                          *
*******************************************************************/
 %let st=%eval(&sex_s+&age_s+&year_s);

        %if &st ^= 0 %then
        %do;

                
        data &DATAWRK..&Condition_File_S;
        merge &DATAWRK..&Condition_File (IN=SET1) &DATAWRK..Person;
        by Person_id;
        year=year(Condition_era_start_date);
        %if &age_s =1 %then %do;
               if year_of_birth NE . then age = year - year_of_birth;
               else age = -999;
                            %end;
        if SET1;
        %if &sex_s = 0 %then %do; drop  Gender_concept_id; %end;
        %if &year_s = 0 %then %do; drop  year; %end;
        drop Year_of_birth;
        run;

            %if &age_s = 1 %then
            %do;

                        data &DATAWRK..all_positive_years;
                        set &DATAWRK..&Condition_File_S;
                        where age >=0;
                        keep age;
                        run;


                        proc univariate data=&DATAWRK..all_positive_years noprint;
                        var age;
                        output out=&DATAWRK..pntls pctlpre=P_ pctlpts= 12.5 to 87.5 by 12.5;
                        run;


                        data _null_;
                        set &DATAWRK..pntls;
                        call symput('PNTL_12_5',p_12_5);
                        call symput('PNTL_25',p_25);
                        call symput('PNTL_37_5',p_37_5);
                        call symput('PNTL_50',p_50);
                        call symput('PNTL_62_5',p_62_5);
                        call symput('PNTL_75',p_75);
                        call symput('PNTL_87_5',p_87_5);
                run;

                        %put &PNTL_12_5 &PNTL_25 &PNTL_37_5 &PNTL_50 &PNTL_62_5 &PNTL_75 &PNTL_87_5;
 
                
                data &DATAWRK..&Condition_File_S;
                        set &DATAWRK..&Condition_File_S;
                        if age < 0 then age_group=0;
                        else if 0<= age <= &PNTL_12_5 then age_group=1;
                else if &PNTL_12_5< age <= &PNTL_25 then age_group=2;   
                else if &PNTL_25< age <= &PNTL_37_5 then age_group=3;
                        else if &PNTL_37_5< age <= &PNTL_50 then age_group=4;
                else if &PNTL_50< age <= &PNTL_62_5 then age_group=5;
                        else if &PNTL_62_5< age <= &PNTL_75 then age_group=6;
                else if &PNTL_75< age <= &PNTL_87_5 then age_group=7;
                        else if &PNTL_87_5< age then age_group=8;
                run;
                        %end;

       
        proc sort data=&DATAWRK..&Condition_File_S
        (keep=
        %if &sex_s = 1  %then  Gender_concept_id;
        %if &age_s = 1  %then  age_group;
        %if &year_s = 1 %then  year;)
        out=&DATAWRK..&Strata NODUPKEY THREADS;
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ; run;

        data &DATAWRK..&Strata; set &DATAWRK..&Strata;
        stratum=_n_; run;

        proc sort data=&DATAWRK..&Condition_File_S THREADS;
        by
        %if &sex_s = 1  %then  Gender_concept_id;
        %if &age_s = 1  %then  age_group;
        %if &year_s = 1 %then year;
        ; run;

        data &DATAWRK..&Condition_File_S;
        merge &DATAWRK..&Condition_File_S &DATAWRK..&Strata;
        by
        %if &sex_s = 1  %then  Gender_concept_id;
        %if &age_s = 1  %then  age_group;
        %if &year_s = 1 %then year;
         ;
        keep Person_id Condition_concept_id Condition_era_start_date Stratum;
        run;
        %end;

        %else
        %do;
                data &DATAWRK..&Strata; stratum=1; run;
                data &DATAWRK..&Condition_File_S; set  &DATAWRK..&Condition_File;
                stratum=1; run;
        %end;




/*******************************************************************
*          Create SRS like drug x condition counts                 *
*******************************************************************/

proc sql THREADS;
create table &DATAWRK..DPTable_1 as
select T2.Drug_concept_id, T1.Condition_concept_id, T1.stratum, count (*) as W00
from  &DATAWRK..&Condition_File_S T1 inner join &DATAWRK..Drug_era T2
ON T1.Person_id=T2.Person_id 
AND T2.Drug_era_start_date<= T1.Condition_era_start_date <= T2.Drug_era_end_date
group by T2.Drug_concept_id, T1.Condition_concept_id, T1.stratum;

create table &DATAWRK..DPTable_2 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, sum(T1.W00)-W00 as W01
from  &DATAWRK..Dptable_1 T1
group by T1.Drug_concept_id, T1.stratum;

create table &DATAWRK..DPTable_3 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, T1.W01,
sum(T1.W00) - W00 as W10
from  &DATAWRK..Dptable_2 T1
group by T1.Condition_concept_id, T1.stratum;

create table &DATAOUT..DP&pvn.&cond_type.2d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, T1.W01,
T1.W10, sum(T1.W00) - T1.W00 - T1.W01 - T1.W10 as W11, sum(T1.W00) as N
from  &DATAWRK..Dptable_3 T1
group by T1.stratum;
quit;
%mend c_s2;


%macro c_s3(datawrk, dataout, cond_type, p_window_drugs, p_window_cond, sex_s, age_s, year_s, dbn, pvn);
/*******************************************************************
*  MACRO c_s3 calculates 3rd counting scenario ("Modified SRS")    *
* for both condition types, prevalent and incident;                *
* INPUT PARAMETERS:                                                *
* datawrk - library that stores working files and preprocessed     *
* data in the cdm format                                           *
* dataout - library for the output file(s)                         *
* cond_type - condition type: 1= prevalent, 2=incident             *
* p_window_drugs - drug persistence window: 0 or 30                *
* p_window_cond - condition persistence window: 0 or 30            *
* sex_s - stratify by sex: 0 or 1                                  *
* age_s - stratify by age: 0 or 1                                  *
* year_s - stratify by year: 0 or 1                                *
* dbn - four letter abbreviation of the database name              *
*******************************************************************/


/*******************************************************************
*          Prepare data for Prevalent/Incident conditions          *
*******************************************************************/
%if &cond_type=1 %then
%do; %let Condition_File=Condition_era; %let Condition_File_S=Condition_era_P;
         %let Strata=Strata_P;
%end;

%if &cond_type=2 %then
%do; %let Condition_File=Condition_era_min; %let Condition_File_S=Condition_era_I;
        %let Strata=Strata_I;
        
        proc sql THREADS;
        create table &DATAWRK..Condition_era_min as
        select T1.Person_id, T1.Condition_concept_id, min(T1.Condition_era_start_date)
        as Condition_era_start_date format DATE9.
        from  &DATAWRK..Condition_era T1
        group by T1.Person_id, T1.Condition_concept_id;
        quit;
        
%end;
/*******************************************************************
*          Stratification                                          *
*******************************************************************/
 %let st=%eval(&sex_s+&age_s+&year_s);

        %if &st ^= 0 %then
        %do;

                
        data &DATAWRK..&Condition_File_S;
        merge &DATAWRK..&Condition_File (IN=SET1) &DATAWRK..Person;
        by Person_id;
        year=year(Condition_era_start_date);
        %if &age_s =1 %then %do;
               if year_of_birth NE . then age = year - year_of_birth;
               else age = -999;
                            %end;
        if SET1;
        %if &sex_s = 0 %then %do; drop  Gender_concept_id; %end;
        %if &year_s = 0 %then %do; drop  year; %end;
        drop Year_of_birth;
        run;

            %if &age_s = 1 %then
            %do;

                        data &DATAWRK..all_positive_years;
                        set &DATAWRK..&Condition_File_S;
                        where age >=0;
                        keep age;
                        run;


                        proc univariate data=&DATAWRK..all_positive_years noprint;
                        var age;
                        output out=&DATAWRK..pntls pctlpre=P_ pctlpts= 12.5 to 87.5 by 12.5;
                        run;


                        data _null_;
                        set &DATAWRK..pntls;
                        call symput('PNTL_12_5',p_12_5);
                        call symput('PNTL_25',p_25);
                        call symput('PNTL_37_5',p_37_5);
                        call symput('PNTL_50',p_50);
                        call symput('PNTL_62_5',p_62_5);
                        call symput('PNTL_75',p_75);
                        call symput('PNTL_87_5',p_87_5);
                run;

                        %put &PNTL_12_5 &PNTL_25 &PNTL_37_5 &PNTL_50 &PNTL_62_5 &PNTL_75 &PNTL_87_5;
 
                
                data &DATAWRK..&Condition_File_S;
                        set &DATAWRK..&Condition_File_S;
                        if age < 0 then age_group=0;
                        else if 0<= age <= &PNTL_12_5 then age_group=1;
                else if &PNTL_12_5< age <= &PNTL_25 then age_group=2;   
                else if &PNTL_25< age <= &PNTL_37_5 then age_group=3;
                        else if &PNTL_37_5< age <= &PNTL_50 then age_group=4;
                else if &PNTL_50< age <= &PNTL_62_5 then age_group=5;
                        else if &PNTL_62_5< age <= &PNTL_75 then age_group=6;
                else if &PNTL_75< age <= &PNTL_87_5 then age_group=7;
                        else if &PNTL_87_5< age then age_group=8;
                run;
                        %end;

       
        proc sort data=&DATAWRK..&Condition_File_S
        (keep=
        %if &sex_s = 1  %then  Gender_concept_id;
        %if &age_s = 1  %then  age_group;
        %if &year_s = 1 %then  year;)
        out=&DATAWRK..&Strata NODUPKEY THREADS;
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ; run;

        data &DATAWRK..&Strata; set &DATAWRK..&Strata;
        stratum=_n_; run;

        proc sort data=&DATAWRK..&Condition_File_S THREADS;
        by
        %if &sex_s = 1  %then  Gender_concept_id;
        %if &age_s = 1  %then  age_group;
        %if &year_s = 1 %then year;
        ; run;

        data &DATAWRK..&Condition_File_S;
        merge &DATAWRK..&Condition_File_S &DATAWRK..&Strata;
        by
        %if &sex_s = 1  %then  Gender_concept_id;
        %if &age_s = 1  %then  age_group;
        %if &year_s = 1 %then year;
         ;
        keep Person_id Condition_concept_id Condition_era_start_date Stratum;
        run;
        %end;

        %else
        %do;
                data &DATAWRK..&Strata; stratum=1; run;
                data &DATAWRK..&Condition_File_S; set  &DATAWRK..&Condition_File;
                stratum=1; run;
        %end;


/*******************************************************************
*          Modified SRS counts                                     *
*******************************************************************/

proc sql THREADS;
create table &DATAWRK..DPTable_1 as
select T2.Drug_concept_id, T1.Condition_concept_id, T1.stratum, count (*) as W00
from  &DATAWRK..&Condition_File_S T1 left join &DATAWRK..Drug_era T2
ON T1.Person_id=T2.Person_id 
AND T2.Drug_era_start_date<= T1.Condition_era_start_date <= T2.Drug_era_end_date
group by T2.Drug_concept_id, T1.Condition_concept_id, T1.stratum;

create table &DATAWRK..DPTable_2 as
select T1.Drug_concept_id, T2.Condition_concept_id, T1.Drug_era_start_date, T1.Person_id, count(*) as W00
from  &DATAWRK..Drug_era T1 left join &DATAWRK..&Condition_File_S T2
ON T1.Person_id=T2.Person_id 
AND T1.Drug_era_start_date<= T2.Condition_era_start_date <= T1.Drug_era_end_date
where T2.Condition_concept_id is missing
group by T1.Drug_concept_id, T2.Condition_concept_id, T1.Drug_era_start_date, T1.Person_id;
quit;

proc sort data=&DATAWRK..DPTable_2 THREADS; by Person_id ;
run;

/*******************************************************************
*          Create strata                                           *
*******************************************************************/
%let st=%eval(&sex_s+&age_s+&year_s);

%if &st ^= 0 %then
%do;
        data &DATAWRK..DPTable_2;
        merge &DATAWRK..DPTable_2 (IN=SET1) &DATAWRK..Person;
        by Person_id;
        year=year(Drug_era_start_date);
        %if &age_s =1 %then
        %do;
        if year_of_birth NE . then age = year - year_of_birth;
        else age = -999;
            if age < 0 then age_group=0;
                        else if 0<= age <= &PNTL_12_5 then age_group=1;
                else if &PNTL_12_5< age <= &PNTL_25 then age_group=2;   
                else if &PNTL_25< age <= &PNTL_37_5 then age_group=3;
                        else if &PNTL_37_5< age <= &PNTL_50 then age_group=4;
                else if &PNTL_50< age <= &PNTL_62_5 then age_group=5;
                        else if &PNTL_62_5< age <= &PNTL_75 then age_group=6;
                else if &PNTL_75< age <= &PNTL_87_5 then age_group=7;
                        else if &PNTL_87_5< age then age_group=8;
        %end;
        
        %if &sex_s = 0 %then %do; drop  Gender_concept_id; %end;
        %if &year_s = 0 %then %do; drop  year; %end;
        %if &age_s = 1 %then %do; drop  age; %end;
        drop Year_of_birth Drug_era_start_date Person_id;
        if SET1;
        run;

        proc sort data=&DATAWRK..DPTable_2 THREADS;
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ;
        run;

        data &DATAWRK..DPTable_2;
        merge &DATAWRK..DPTable_2 (IN=SET1) &DATAWRK..&Strata (IN=SET2);
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ;
        if SET1 AND SET2;
        keep Drug_concept_id Condition_concept_id W00 Stratum;
        run;
%end;
        %else
%do;
        data &DATAWRK..DPTable_2; set &DATAWRK..DPTable_2;
        stratum=1; run;
%end;

/*******************************************************************
*          for "null_condition": Condition_concept_id=-999         *
*          for "null_drug": Drug_concept_id = -999                *
*******************************************************************/
data  &DATAWRK..DPTable_2;
set  &DATAWRK..DPTable_2;
if Condition_concept_id=. then Condition_concept_id=-999;
run;

proc sql THREADS;
create table &DATAWRK..DPtable_3 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, sum(T1.W00) as W00
from &DATAWRK..DPtable_2 T1
group by T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum;
quit;

data  &DATAWRK..DPTable_3;
set  &DATAWRK..DPTable_1 &DATAWRK..DPTable_3;
if Drug_concept_id=. then Drug_concept_id=-999;
run;

proc sql THREADS;
create table &DATAWRK..DPTable_4 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, sum(T1.W00)-W00 as W01
from  &DATAWRK..Dptable_3 T1
group by T1.Drug_concept_id, T1.stratum;

create table &DATAWRK..DPTable_5 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, T1.W01,
sum(T1.W00) - W00 as W10
from  &DATAWRK..Dptable_4 T1
group by T1.Condition_concept_id, T1.stratum;


create table &DATAOUT..DP&pvn.&cond_type.3d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s  as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, T1.W00, T1.W01,
T1.W10, sum(T1.W00) - T1.W00 - T1.W01 - T1.W10 as W11, sum(T1.W00) as N
from  &DATAWRK..Dptable_5 T1
group by T1.stratum;
quit;
%mend c_s3;

%macro c_s4(datawrk, dataout, cond_type, p_window_drugs, p_window_cond, sex_s, age_s, year_s, dbn, pvn);

/*******************************************************************
*  MACRO c_s4 calculates 4th counting scenario ("Curtis")          *
* for both condition types, prevalent and incident;                *
* INPUT PARAMETERS:                                                *
* datawrk - library that stores working files and preprocessed     *
* data in the cdm format                                           *
* dataout - library for the output file(s)                         *
* cond_type - condition type: 1= prevalent, 2=incident             *
* p_window_drugs - drug persistence window: 0 or 30                *
* p_window_cond - condition persistence window: 0 or 30            *
* sex_s - stratify by sex: 0 or 1                                  *
* age_s - stratify by age: 0 or 1                                  *
* year_s - stratify by year: 0 or 1                                *
* dbn - four letter abbreviation of the database name              *
*******************************************************************/

/*******************************************************************
*          Prepare data for Prevalent/Incident conditions          *
*******************************************************************/
%if &cond_type=1 %then
%do; %let Condition_File=Condition_era; %let Condition_File_S=Condition_era_P;
         %let Strata=Strata_P;
%end;

%if &cond_type=2 %then
%do; %let Condition_File=Condition_era_min; %let Condition_File_S=Condition_era_I;
        %let Strata=Strata_I;
        
                proc sql;
                create table &DATAWRK..Condition_era_min as
                select DISTINCT T1.Person_id, T1.Condition_concept_id, min(T1.Condition_era_start_date)
                as Condition_era_start_date format DATE9.
                from  &DATAWRK..Condition_era T1
                group by T1.Person_id, T1.Condition_concept_id;
                quit;
      
%end;
/*******************************************************************
*          Stratification                                          *
*******************************************************************/
 %let st=%eval(&sex_s+&age_s+&year_s);

 %if &st ^= 0 %then
 %do;

        data &DATAWRK..&Condition_File_S;
        merge &DATAWRK..&Condition_File (IN=SET1) &DATAWRK..Person;
        by Person_id;
        year=year(Condition_era_start_date);
        %if &age_s =1 %then %do;
                if year_of_birth NE . then age = year - year_of_birth;
                else age = -999;
            %end;
        if SET1;
        %if &sex_s = 0 %then %do; drop  Gender_concept_id; %end;
        %if &year_s = 0 %then %do; drop  year; %end;
               drop Year_of_birth;
        run;
                        %if &age_s = 1 %then
            %do;

                        data &DATAWRK..all_positive_years;
                        set &DATAWRK..&Condition_File_S;
                        where age >=0;
                        keep age;
                        run;


                        proc univariate data=&DATAWRK..all_positive_years noprint;
                        var age;
                        output out=&DATAWRK..pntls pctlpre=P_ pctlpts= 12.5 to 87.5 by 12.5;
                        run;


                        data _null_;
                        set &DATAWRK..pntls;
                        call symput('PNTL_12_5',p_12_5);
                        call symput('PNTL_25',p_25);
                        call symput('PNTL_37_5',p_37_5);
                        call symput('PNTL_50',p_50);
                        call symput('PNTL_62_5',p_62_5);
                        call symput('PNTL_75',p_75);
                        call symput('PNTL_87_5',p_87_5);
                run;

                        %put &PNTL_12_5 &PNTL_25 &PNTL_37_5 &PNTL_50 &PNTL_62_5 &PNTL_75 &PNTL_87_5;
 
                
                data &DATAWRK..&Condition_File_S;
                        set &DATAWRK..&Condition_File_S;
                        if age < 0 then age_group=0;
                        else if 0<= age <= &PNTL_12_5 then age_group=1;
                else if &PNTL_12_5< age <= &PNTL_25 then age_group=2;   
                else if &PNTL_25< age <= &PNTL_37_5 then age_group=3;
                        else if &PNTL_37_5< age <= &PNTL_50 then age_group=4;
                else if &PNTL_50< age <= &PNTL_62_5 then age_group=5;
                        else if &PNTL_62_5< age <= &PNTL_75 then age_group=6;
                else if &PNTL_75< age <= &PNTL_87_5 then age_group=7;
                        else if &PNTL_87_5< age then age_group=8;
                run;
                        %end;

                proc sort data=&DATAWRK..&Condition_File_S
                (keep=
                %if &sex_s = 1  %then  Gender_concept_id;
                %if &age_s = 1  %then  age_group;
                %if &year_s = 1 %then  year;)
                out=&DATAWRK..&Strata NODUPKEY;
                by
                %if &sex_s = 1 %then  Gender_concept_id;
                %if &age_s = 1 %then  age_group;
                %if &year_s = 1 %then year;
                ; run;

            data &DATAWRK..&Strata; set &DATAWRK..&Strata;
        stratum=_n_; run;

                proc sort data=&DATAWRK..&Condition_File_S;
                by
                %if &sex_s = 1  %then  Gender_concept_id;
                %if &age_s = 1  %then  age_group;
                %if &year_s = 1 %then year;
                ; run;

                data &DATAWRK..&Condition_File_S;
                merge &DATAWRK..&Condition_File_S &DATAWRK..&Strata;
                by
                %if &sex_s = 1  %then  Gender_concept_id;
                %if &age_s = 1  %then  age_group;
                %if &year_s = 1 %then year;
                ;
                keep Person_id Condition_concept_id Condition_era_start_date Stratum;
                run;
        %end;

        %else
        %do;
                data &DATAWRK..&Strata; stratum=1; run;
                data &DATAWRK..&Condition_File_S; set  &DATAWRK..&Condition_File;
                stratum=1; run;
        %end;

/*******************************************************************
*          Prepare Drug eras measured in months                    *
*******************************************************************/
data &datawrk..Drug_era_M;
set &datawrk..Drug_era;
drug_era_start_month = (12*(year(drug_era_start_date)-1960))+month(drug_era_start_date);
drug_era_end_month = (12*(year(drug_era_end_date)-1960))+month(drug_era_end_date);
keep Person_id Drug_concept_id drug_era_start_month drug_era_end_month;
run;

/*******************************************************************
*          Re-write Drug eras (one month - one record)             *
*******************************************************************/
data &datawrk..Drug_by_month;
set &datawrk..Drug_era_M;
do month=Drug_era_start_month to Drug_era_end_month;
output;
end;
drop Drug_era_start_month Drug_era_end_month;
run;

/*******************************************************************
*          Prepare Condition eras measured in months               *
*******************************************************************/
data &datawrk..Condition_era_M;
set &datawrk..&Condition_File_S;
condition_era_start_month = (12*(year(Condition_era_start_date)-1960))+month(Condition_era_start_date);
keep Person_id Condition_concept_id Condition_era_start_month Stratum;
run;

/*******************************************************************
*          Count (drug x condition) pairs                          *
*******************************************************************/
proc sql;
create table &DATAWRK..DPTable_1DC as
select T1.Drug_concept_id, T2.Condition_concept_id, T2.Stratum, count(*) as W00
from  &DATAWRK..Drug_era_M T1, &DATAWRK..Condition_era_M T2
where T1.Person_id=T2.Person_id AND ((T1.Drug_era_start_month<= T2.Condition_era_start_month <= T1.Drug_era_end_month)
OR (T1.Drug_era_start_month<= (T2.Condition_era_start_month-1) <= T1.Drug_era_end_month))
group by T1.Drug_concept_id, T2.Condition_concept_id, T2.Stratum;
quit;

/*******************************************************************
*          Count (drug x _no_condition_) pairs                     *
*******************************************************************/
proc sql THREADS;
create table &DATAWRK..DPTable_1DC_1 as
select T1.Drug_concept_id, T2.Condition_concept_id, T1.month, T1.Person_id, count(*) as W00
from  &DATAWRK..Drug_by_month T1 left join &DATAWRK..Condition_era_M T2
ON T1.Person_id=T2.Person_id AND ((T1.month = T2.Condition_era_start_month)
OR (T1.month = (T2.Condition_era_start_month - 1)))
where T2.Condition_concept_id is missing
group by T1.Drug_concept_id, T2.Condition_concept_id, T1.month, T1.Person_id;
quit;

/*******************************************************************
*     Add stratification variable to (drug x _no_condition_) pairs *
*******************************************************************/
proc sort data=&DATAWRK..DPTable_1DC_1;
by Person_id;
run;

%let st=%eval(&sex_s+&age_s+&year_s);
%if &st ^= 0 %then
%do;
        data &DATAWRK..DPTable_1DC_1;
        merge &DATAWRK..DPTable_1DC_1 (IN=SET1) &DATAWRK..Person;
        by Person_id;
        year=month/12 +1960;
        %if &age_s =1 %then %do;
        if year_of_birth NE . then age = year - year_of_birth;
        else age = -999;
            if age < 0 then age_group=0;
                        else if 0<= age <= &PNTL_12_5 then age_group=1;
                else if &PNTL_12_5< age <= &PNTL_25 then age_group=2;   
                else if &PNTL_25< age <= &PNTL_37_5 then age_group=3;
                        else if &PNTL_37_5< age <= &PNTL_50 then age_group=4;
                else if &PNTL_50< age <= &PNTL_62_5 then age_group=5;
                        else if &PNTL_62_5< age <= &PNTL_75 then age_group=6;
                else if &PNTL_75< age <= &PNTL_87_5 then age_group=7;
                        else if &PNTL_87_5< age then age_group=8;
        %end;

        if SET1;
        %if &sex_s = 0 %then %do; drop  Gender_concept_id; %end;
        %if &year_s = 0 %then %do; drop  year; %end;
        %if &age_s = 1 %then %do; drop  age; %end;
        drop Year_of_birth month Person_id;
        if SET1;
        run;

        proc sort data=&DATAWRK..DPTable_1DC_1;
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ; run;

        data &DATAWRK..DPTable_1DC_1;
        merge &DATAWRK..DPTable_1DC_1 (IN=SET1) &DATAWRK..&Strata (IN=SET2);
        by
        %if &sex_s = 1 %then  Gender_concept_id;
        %if &age_s = 1 %then  age_group;
        %if &year_s = 1 %then year;
        ;
        if SET1 AND SET2;
        keep Drug_concept_id Condition_concept_id W00 Stratum;
        run;
%end;
%else
%do;
        data &DATAWRK..DPTable_1DC_1;
        set &DATAWRK..DPTable_1DC_1;
        stratum=1; run;
%end;

proc sql THREADS;
create table &DATAWRK..DPTable_1DC_1_ as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum, sum(T1.W00) as W00
from &DATAWRK..DPTable_1DC_1 T1
group by T1.Drug_concept_id, T1.Condition_concept_id, T1.stratum;
quit;



/*******************************************************************
*          Count (_no_drug_ x condition) pairs                     *
*******************************************************************/
proc sql;
create table &DATAWRK..DPTable_1DC_2 as
select T2.Drug_concept_id, T1.Condition_concept_id, T1.Stratum, count(*) as W00
from  &DATAWRK..Condition_era_M T1 left join  &DATAWRK..Drug_era_M T2
ON T1.Person_id=T2.Person_id AND ((T2.Drug_era_start_month<= T1.Condition_era_start_month <= T2.Drug_era_end_month)
OR (T2.Drug_era_start_month<= (T1.Condition_era_start_month-1) <= T2.Drug_era_end_month))
where T2.Drug_concept_id is missing
group by T2.Drug_concept_id, T1.Condition_concept_id, T1.stratum;
quit;


data &DATAWRK..DPTable_1DC_all;
set &DATAWRK..DPTable_1DC &DATAWRK..DPTable_1DC_1_ &DATAWRK..DPTable_1DC_2;
if Condition_concept_id=. then Condition_concept_id=-999;
if Drug_concept_id=. then Drug_concept_id=-999;
run;

/*******************************************************************
*          Produce final file with counts                          *
*******************************************************************/
proc sql;
create table &DATAWRK..DPTable_4 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.Stratum, T1.W00, sum(T1.W00)-W00 as W01
from  &DATAWRK..DPTable_1DC_all T1
group by T1.Drug_concept_id, T1.Stratum;

create table &DATAWRK..DPTable_5 as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.Stratum, T1.W00, T1.W01,
sum(T1.W00) - W00 as W10
from  &DATAWRK..Dptable_4 T1
group by T1.Condition_concept_id, T1.Stratum;


create table &DATAWRK..DP&pvn.&cond_type.4d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s  as
select T1.Drug_concept_id, T1.Condition_concept_id, T1.Stratum, T1.W00, T1.W01,
T1.W10, sum(T1.W00) - T1.W00 - T1.W01 - T1.W10 as W11, sum(T1.W00) as N
from  &DATAWRK..Dptable_5 T1
group by T1.stratum;
quit;

%mend c_s4;


/*******************************************************************
*          DP Metrics dpm_1, dpm_2,..., dpm_11                     *
*******************************************************************/
%macro dpm_1(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  PRR macro                                                                                   *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/

proc sql;
create table &dataout..&filein.m1 as
select T1.Drug_concept_id, T1.Condition_concept_id, sum(W00*(W10 + W11)/N)/sum(W10*(W00 + W01)/N) as score
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;

data &dataout..&filein.m1;
set &dataout..&filein.m1;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m1;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m1;
        merge &dataout..&filein.m1 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;

%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m1.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m1   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_1;

%macro dpm_2(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  ROR macro                                                                                                       *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/
proc sql;
create table &dataout..&filein.m2 as
select T1.Drug_concept_id, T1.Condition_concept_id, sum(W00*W11/N)/sum(W10*W01/N) as score
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;

data &dataout..&filein.m2;
set &dataout..&filein.m2;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m2;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m2;
        merge &dataout..&filein.m2 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;



data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m2.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m2   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_2;

%macro dpm_3(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  BCPNN macro,                                                    *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/

proc sql;
create table &dataout..&filein.m3 as
select T1.Drug_concept_id, T1.Condition_concept_id,
LOG2((sum(W00)+0.5)/(sum((W00+W01)*(W00+W10)/N)+0.5)) as score
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;

data &dataout..&filein.m3;
set &dataout..&filein.m3;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m3;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m3;
        merge &dataout..&filein.m3 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m3.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m3   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_3;


%macro dpm_4(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  EBGM macro,                                                     *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/

proc sql THREADS;
create table &datain..&filein.EBGM1 as
select T1.Drug_concept_id, T1.Condition_concept_id, sum(T1.W00) as N_ij,
sum((W00+ W01)*(W00+W10)/N) as E_ij
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;




proc iml;
/*******************************************************************
*   Optimization procedure for a mixture of                        *
*   negative binomial densities                                    *
*******************************************************************/


USE &datain..&filein.EBGM1;  READ all var{N_ij E_ij}; CLOSE &datain..&filein.EBGM1;
MAX_N=max(N_ij); m=nrow(N_ij);


/*******************************************************************
*   OBJECTIVE FUNCTION                                             *
*******************************************************************/
start NEGBIN_I(x) global(N_ij, E_ij, MAX_N, m);

x1=x[1]; x2=x[2]; x3=x[3]; x4=x[4]; x5=x[5];

GNB1=J(MAX_N,1,0); GNB3=J(MAX_N,1,0); GS1=J(MAX_N,1,0);
GS3=J(MAX_N,1,0); G2S1=J(MAX_N,1,0); G2S3=J(MAX_N,1,0);

pro1=1; pro3=1;
sum_1=0; sum_3=0; cum_1=0; cum_3=0;

do i=1 to MAX_N by 1;
        pro1=pro1*((x1-1)/i+1);
        pro3=pro3*((x3-1)/i+1);
        GNB1[i]=pro1;
        GNB3[i]=pro3;
        sum_1=sum_1+1/(i+x1-1);
        sum_3=sum_3+1/(i+x3-1);
        GS1[i]=sum_1;
        GS3[i]=sum_3;
        cum_1=cum_1+1/((i+x1-1)**2);
        cum_3=cum_3+1/((i+x3-1)**2);
        G2S1[i]=cum_1;
        G2S3[i]=cum_3;
end;

p1=E_ij/(E_ij+x2); p2=E_ij/(E_ij+x4);
q1=1-p1; q2=1-p2;


GNBA1=GNB1[N_ij];
GNBA3=GNB3[N_ij];
fa1=GNBA1#(p1##N_ij)#(q1##x1);
fa2=GNBA3#(p2##N_ij)#(q2##x3);
fa=x5#fa1+(1-x5)#fa2;

mc=1.e-150; c=(fa >mc); c1=1-c;
fa=fa#c + c1#mc; fa=log(fa);
y1=sum(fa);
return(y1);
finish NEGBIN_I;


/*******************************************************************
*   DERIVATIVE/GRADIENT                                            *
*******************************************************************/
start G_NEGBIN_I(x) global(N_ij, E_ij, MAX_N, m);

g=j(1,5,0);
x1=x[1]; x2=x[2]; x3=x[3]; x4=x[4]; x5=x[5];

GNB1=J(MAX_N,1,0); GNB3=J(MAX_N,1,0); GS1=J(MAX_N,1,0);
GS3=J(MAX_N,1,0); G2S1=J(MAX_N,1,0); G2S3=J(MAX_N,1,0);

pro1=1; pro3=1;
sum_1=0; sum_3=0; cum_1=0; cum_3=0;

do i=1 to MAX_N by 1;
        pro1=pro1*((x1-1)/i+1);
        pro3=pro3*((x3-1)/i+1);
        GNB1[i]=pro1;
        GNB3[i]=pro3;
        sum_1=sum_1+1/(i+x1-1);
        sum_3=sum_3+1/(i+x3-1);
        GS1[i]=sum_1;
        GS3[i]=sum_3;
        cum_1=cum_1+1/((i+x1-1)**2);
        cum_3=cum_3+1/((i+x3-1)**2);
        G2S1[i]=cum_1;
        G2S3[i]=cum_3;
end;

p1=E_ij/(E_ij+x2); p2=E_ij/(E_ij+x4);
q1=1-p1; q2=1-p2;
mc=1.e-150;
cq1=(q1 > 0); cq11=1-cq1;
q1=q1#cq1 + cq11#mc;
cq2=(q2 > 0); cq21=1-cq2;
q2=q2#cq2 + cq21#mc;


GNBA1=GNB1[N_ij]; GNBA3=GNB3[N_ij];
fa1=GNBA1#(p1##N_ij)#(q1##x1);
fa2=GNBA3#(p2##N_ij)#(q2##x3);
fa=x5#fa1+(1-x5)#fa2;

mc=1.e-150; c=(fa >mc); c1=1-c;
fa=fa#c + c1#mc;

r2a=GS1[N_ij]+log(q1); f_by_alpha12=fa1#r2a;
sum1=sum(x5#f_by_alpha12/fa); r1a=fa1#p1#q1;
r2a2=-1#N_ij/(E_ij#q1)+x1#E_ij/(x2#x2#p1);
f_by_beta12=r1a#r2a2;
sum2=sum(x5#f_by_beta12/fa);

r2a3=GS3[N_ij]+log(q2); f_by_alpha34=fa2#r2a3;
sum3=sum((1-x5)#f_by_alpha34/fa);

r1a4=fa2#p2#q2; r2a4=-1#N_ij/(E_ij#q2)+x3#E_ij/(x4#x4#p2);
f_by_beta34=r1a4#r2a4;
sum4=sum((1-x5)#f_by_beta34/fa);
sum5=sum((fa1-fa2)/fa);

g[1]=sum1; g[2]=sum2; g[3]=sum3; g[4]=sum4; g[5]=sum5;
return(g);
finish G_NEGBIN_I;


/*******************************************************************
*   HESSIAN                                                        *
*******************************************************************/
start H_NEGBIN_I(x) global(N_ij, E_ij, MAX_N, m);

hes=j(5,5,0);
x1=x[1]; x2=x[2]; x3=x[3]; x4=x[4]; x5=x[5];

GNB1=J(MAX_N,1,0); GNB3=J(MAX_N,1,0); GS1=J(MAX_N,1,0);
GS3=J(MAX_N,1,0); G2S1=J(MAX_N,1,0); G2S3=J(MAX_N,1,0);

pro1=1; pro3=1;
sum_1=0; sum_3=0; cum_1=0; cum_3=0;

do i=1 to MAX_N by 1;
        pro1=pro1*((x1-1)/i+1);
        pro3=pro3*((x3-1)/i+1);
        GNB1[i]=pro1;
        GNB3[i]=pro3;
        sum_1=sum_1+1/(i+x1-1);
        sum_3=sum_3+1/(i+x3-1);
        GS1[i]=sum_1;
        GS3[i]=sum_3;
        cum_1=cum_1+1/((i+x1-1)**2);
        cum_3=cum_3+1/((i+x3-1)**2);
        G2S1[i]=cum_1;
        G2S3[i]=cum_3;
end;
onex5=1-x5;

p1=E_ij/(E_ij+x2); p2=E_ij/(E_ij+x4);
q1=1-p1; q2=1-p2;
mc=1.e-150;
cq1=(q1 > 0); cq11=1-cq1;
q1=q1#cq1 + cq11#mc;
cq2=(q2 > 0); cq21=1-cq2;
q2=q2#cq2 + cq21#mc;

GNBA1=GNB1[N_ij]; GNBA3=GNB3[N_ij];
GS1A=GS1[N_ij]; G2S1A=G2S1[N_ij];
GS3A=GS3[N_ij]; G2S3A=G2S3[N_ij];

fa1=GNBA1#(p1##N_ij)#(q1##x1);
fa2=GNBA3#(p2##N_ij)#(q2##x3);
fa=x5#fa1+(1-x5)#fa2;

mc=1.e-150; c=(fa >mc); c1=1-c;
fa=fa#c + c1#mc;
delf12 = fa1 - fa2;
den=fa##2;

r2_aa1=GS1A + log(q1); fbyaa12=fa1#(r2_aa1##2- G2S1A);
r2_aa2=GS3A+log(q2); fbyaa34=fa2#(r2_aa2##2-G2S3A);

r1_ab1=fa1#p1; r3_ab1=(x1/x2-N_ij/E_ij);
fbyab12=r1_ab1#(1/x2+r2_aa1#r3_ab1);

r1_ab2=fa2#p2; r3_ab2=(x3/x4-N_ij/E_ij);
fbyab34=r1_ab2#(1/x4+r2_aa2#r3_ab2);

fbyalpha12=fa1#r2_aa1; fbyalpha34=fa2#r2_aa2;

r1_beta1=r1_ab1#q1; r2_beta1=-1#N_ij/(E_ij#q1)+x1#E_ij/(x2#x2#p1);
fbybeta12=r1_beta1#r2_beta1;

r1_beta2=r1_ab2#q2; r2_beta2=-1#N_ij/(E_ij#q2)+x3#E_ij/(x4#x4#p2);
fbybeta34=r1_beta2#r2_beta2;

r2=1/(x2#(E_ij+x2)); r3=(x1#E_ij - x2#N_ij)/(E_ij+x2);
r4=(x1#E_ij)/x2 - N_ij; r5=N_ij#(x2##2)-2#x1#x2#E_ij - x1#(E_ij##2);
fbybb12=fa1#r2#(r3#r4+r5#r2);

r2=1/(x4#(E_ij+x4)); r3=(x3#E_ij-x4*N_ij)/(E_ij+x4);
r4=(x3#E_ij)/x4-N_ij;
r5=N_ij#(x4##2)-2#x3#x4#E_ij-x3#(E_ij##2);
fbybb34=fa2#r2#(r3#r4+r5#r2);

sum11 = sum((x5#fbyaa12#fa-(x5#fbyalpha12)##2)/den);
sum12 = sum(x5#(fbyab12#fa - fbyalpha12#x5#fbybeta12)/den);
sum13 = -1*sum(x5#fbyalpha12#onex5#fbyalpha34/den);
sum14 = -1*sum(x5#fbyalpha12#onex5#fbybeta34/den);
sum15 = sum((fbyalpha12#fa - x5#fbyalpha12#delf12)/den);
sum22 = sum((x5#fbybb12#fa-(x5#fbybeta12)##2)/den);
sum23 = -1*sum(x5#fbybeta12#onex5#fbyalpha34/den);
sum24 = -1*sum(x5#fbybeta12#onex5#fbybeta34/den);
sum25 = sum((fbybeta12#fa-x5#fbybeta12#delf12)/den);
sum33 = sum((onex5#fbyaa34#fa - (onex5#fbyalpha34)##2)/den);
sum34 = sum(onex5#(fbyab34#fa - fbyalpha34#onex5#fbybeta34)/den);
sum35 = -1*sum((fbyalpha34#fa + onex5#fbyalpha34#delf12)/den);
sum44 = sum((onex5#fbybb34#fa - (onex5#fbybeta34)##2)/den);
sum45 = -1*sum((fbybeta34#fa + onex5#fbybeta34#delf12)/den);
sum55 = -1*sum((delf12##2)/den);

hes[1,1]=sum11; hes[1,2]=sum12; hes[1,3]=sum13; hes[1,4]=sum14; hes[1,5]=sum15;
hes[2,1]=sum12; hes[2,2]=sum22; hes[2,3]=sum23; hes[2,4]=sum24; hes[2,5]=sum25;
hes[3,1]=sum13; hes[3,2]=sum23; hes[3,3]=sum33; hes[3,4]=sum34; hes[3,5]=sum35;
hes[4,1]=sum14; hes[4,2]=sum24; hes[4,3]=sum34; hes[4,4]=sum44; hes[4,5]=sum45;
hes[5,1]=sum15; hes[5,2]=sum25; hes[5,3]=sum35; hes[5,4]=sum45; hes[5,5]=sum55;
return(hes);
finish H_NEGBIN_I;


/*******************************************************************
*   x0 - initial value for optimization procedure                  *
*******************************************************************/
x0={0.2 0.1 2 4 0.3333};

y1=NEGBIN_I(x0);
print y1;

y2=G_NEGBIN_I(x0);
print y2;

y3=H_NEGBIN_I(x0);
print y3;


con={1.e-6 1.e-6 1.e-6 1.e-6 0,
. . . . 1};
optn={1 2};
par={. 0.5 . . . . . . . .};
call nlpnra(rc,xres,"NEGBIN_I",x0,optn,con,,par,,"G_NEGBIN_I","H_NEGBIN_I");

print rc;

a1=xres[1]; b1=xres[2]; a2=xres[3]; b2=xres[4]; P=xres[5];
GNB1=J(MAX_N,1,0); GNB3=J(MAX_N,1,0);

pro1=1; pro3=1;

do i=1 to MAX_N by 1;
        pro1=pro1*((a1-1)/i+1);
        pro3=pro3*((a2-1)/i+1);
        GNB1[i]=pro1;
        GNB3[i]=pro3;
end;

p1=E_ij/(E_ij+b1); p2=E_ij/(E_ij+b2);
q1=1-p1; q2=1-p2;
GNBA1=GNB1[N_ij];
GNBA3=GNB3[N_ij];
fa1=GNBA1#(p1##N_ij)#(q1##a1);
fa2=GNBA3#(p2##N_ij)#(q2##a2);
fa=P#fa1+(1-P)#fa2;

mc=1.e-150; c=(fa >mc); c1=1-c;
fa=fa#c + c1#mc;
Qn=P#fa1/fa;

n1=N_ij+a1; n2=N_ij+a2;
e1=E_ij+b1; e2=E_ij+b2;

eln_1=digamma(n1)-log(e1);
eln_2=digamma(n2)-log(e2);

ELN=Qn#eln_1+(1-Qn)#eln_2;
EBL2=ELN/log(2); EBGM=2##EBL2;

create &datain..a1 from a1[colname='a1']; append from a1; close &datain..a1;
create &datain..a2 from a2[colname='a2']; append from a2; close &datain..a2;
create &datain..b1 from b1[colname='b1']; append from b1; close &datain..b1;
create &datain..b2 from b2[colname='b2']; append from b2; close &datain..b2;
create &datain..P from P[colname='P']; append from P; close &datain..P;
create &datain..ebgm from ebgm[colname='score']; append from ebgm; close &datain..ebgm;

data &datain..param_out; merge &datain..a1 &datain..b1 &datain..a2 &datain..b2 &datain..p;
run;

data &dataout..&filein.m4;
merge &datain..&filein.EBGM1 &datain..EBGM;
drop N_ij E_ij; run;
quit;

data &dataout..&filein.m4;
set &dataout..&filein.m4;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m4;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m4;
        merge &dataout..&filein.m4 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m4.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m4   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_4;

%macro dpm_5(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  Gould's method,                                                 *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/


proc sql THREADS;
create table &datain..&filein.U1 as
select T1.Drug_concept_id, T1.Condition_concept_id, sum(T1.W00) as N,
sum((W00+ W01)*(W00+W10)/N) as E
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;

proc sql THREADS;
create table &datain..&filein.U2 as
select T1.N, T1.E, count(*) as wt
from  &datain..&filein.U1 T1
group by T1.N, T1.E;
quit;

%****************************************************************;
%*  MACRO NAME: BayesCalcs.sas                                            *;
%*  FUNCTION: perform calculations required for a Bayesian Analysis of a  *;
%*            collection of observed/expected drug-event report counts.   *;
%*  AUTHOR: Lawrence Gould                                                *;
%*     PROGRAMMER: Shuping Zhang (managed by Amy Gillespie)               *;
%* DATE COMPLETED: 26FEB2008 (translate R code to SAS)                    *;
%*      REVISIONS:                                                        *;
%*    SAS VERSION: 9.13 or higher                                         *;
%*  PLATFORM USED: Microsoft Windows XP                                   *;
%*    ASSUMPTIONS: the SAS PROC IML has been ready.                       *;
%*     INPUT DATA: refer the macro parameters below.                      *;
%*   MACRO OUTPUT: pargs, PstDen_p, ProbH0, CDFpsLm, LmbdCrit, BFpost     *;
%*                 pargs and PstDen_p are saved in SAS dataset BayesOut1, *;
%*                 all others are saved in SAS dataset BayesOut2.         *;
%*        pargs =  a vector of the values of p for which values of the    *;
%*                 posterior density of p was computed (numpargs elements)*;
%*     PstDen_p =  the values of the posterior density of p evaluated for *;
%*                 the arguments given in pargs (has numpargs elements).  *;
%*       ProbH0 =  a vector of values of the posterior probability that   *;
%*                 the lambda parameter for a drug-event pair was drawn   *;
%*                 from the 'null' prior distribution, with parameters    *;
%*                 (a0, b0).                                              *;
%*       BFpost =  Bayes factor for hypothesis lambda > lambda0           *;
%*              = [1-CDFpost(lambda)]/CDFpost(lambda) divided by          *;
%*                [1-CDFpre(lambda) ]/CDFpre(lambda) calculated for each  *;
%*                drug-event pair and each value in the input vector      *;
%*                'lambdas'.                                              *;
%*     CDFpsLm =  value of the posterior cdf of lambda evaluated          *;
%*                at each value in the input vector 'lambdas'.            *;
%*     LmbdCrit = for each drug-event pair the value of lambda            *;
%*                such that the posterior cdf of lambda                   *;
%*                evaluated for that drug-event pair = 'targ_p'.          *;
%**************************************************************************;

%macro BayesCalcs(
 N =        /* vector of observed drug-event report counts.                */
,E =        /* vector of expected drug-event report counts.                */
/*              Assume that N,E pairs are unique                           */
,wts =      /* vector of counts of unique N, E pairs                       */
,nullprior =  /* parameters (a0, b0) of prior distn of lambda under 'null'.*/
,altprior = /* parameters (a1, b1) of prior distn of lambda under 'alt.'.  */
,numpargs = /* #of values p at which the posterior density is              */
            /* to be computed.                                             */
,lambdas = /* values at which posterior cdf of lambda is to be calculated. */
,alpha  =  /* parameter of prior density for p.                            */
,targ_p = /* value at which to invert the posterior cdf of lambda          */
);

%*--------------------------------------------------------------*;
%* set macro parameters default if user leaves them blank.      *;
%*--------------------------------------------------------------*;
  %if "&numpargs" eq "" %then %let numpargs = 100;
  %if "&lambdas" eq "" %then %let lambdas = {1,2,4};
  %if "&alpha" eq "" %then %let alpha = 3;
  %if "&targ_p" eq "" %then %let targ_p = 0.05;

%*-------------------------------------------------------------------------*;
%* Get the mode of the posterior density of p, also limits of range over   *;
%* which density is "nonzero". Then calculate "raw" density over the range.*;
%*-------------------------------------------------------------------------*;
  %getmode(N=&N, E=&E, wts=&wts, nullprior=&nullprior,
           altprior=&altprior, alpha=&alpha);

  pargs = do(pL,pU,(pU-pL)/&numpargs)`;
                    %* p<-seq(y$pL,y$pU,length.out = numpargs+1);
  v = &alpha*log(pargs);
  do ii = 1 to nrow(pargs);
    v[ii] = v[ii] + &wts`*log(pargs[ii]*f1 + (1-pargs[ii])*f2);
  end;
  v = exp(v-max(v));
%*--------------------------------------------------------------*;
%* Steps for integration using Simpson-rule to get normalizing  *;
%* constant for posterior density of p.                         *;
%*--------------------------------------------------------------*;
  a1 = &nullprior.[1]+&N;  b01 = &nullprior.[2];  b1 = b01+&E;
%*        q01 = b01/b1;
  a2 = &altprior.[1] +&N;  b02 = &altprior.[2];   b2 = b02+&E;
%*        q02 = b02/b2;
  nlam = nrow(&lambdas);
  np2 = &numpargs/2;

  cc = {1}||repeat(0,1,&numpargs-1)||{1};
%*                                 cc<-c(1,array(0,numpargs-1),1);
  cc[2*(1:np2)] = 4;                    %* cc[2*(1:np2)]<-4;
  cc[2*(1:(np2-1))+1] = 2;              %* cc[2*(1:(np2-1))+1]<-2;
  cc = (pargs[2]-pargs[1])*cc/3;
  qq = cc*v;               %* qq<-cc %*% v (normalizing constant);
  PstDen_p = v/qq;         %* marginal posterior density of p;
%*--------------------------------------------------------------*;
%* Set up for another Simpson-rule integration.                 *;
%*--------------------------------------------------------------*;
  w = PstDen_p#cc`;                        %* w<-cc*PostDen.p;
  pp = (1-pargs)/pargs;
  ProbH0 = J(nrow(f1),1,0);
  do ii = 1 to nrow(pp);
    ProbH0 = ProbH0 + w[ii]/(1 + pp[ii]#f2/f1);
  end;
/*
  do ii = 1 to nrf;
    qq = 1/(1 + (f2[ii]/f1[ii])#pp);
                    %* qq<-1/(1+outer(y$f2/y$f1,(1-p)/p,"*"));
    ProbH0[ii] = w`*qq;                  %* ProbH0<- qq %*% w;
  end;
*/
%*------------------------------------------------------------------------*;
%* find the argument where the posterior CDF of lambda equals the targ_p. *;
%*------------------------------------------------------------------------*;
   *print a1 b1 ;

  zp  = probit(&targ_p);;
  do i = 1 to nrow(&N);
    if (a1[i] < 1000) then lower = gaminv(&targ_p,a1[i])/b1[i];
      else lower = 0.25*(zp + sqrt(4*a1[i]-1))**2/b1[i];
    if (a2[i] < 1000) then upper = gaminv(&targ_p,a2[i])/b2[i];
      else upper = 0.25*(zp + sqrt(4*a2[i]-1))**2/b2[i];
    tel = 0;
    do while
       ( round(abs(upper-lower),.0000001)>0.0000001 & tel<10000 );
      tel = tel+1;
      x = (lower+upper)/2;
      fn = ProbH0[i]*cdf('GAMMA',x,a1[i],1/b1[i])
           +(1-ProbH0[i])*cdf('GAMMA',x,a2[i],1/b2[i])-&targ_p;
      if fn = 0 then do; tel = 10000; end; else
      if fn < 0 then do; lower = x;   end; else
      if fn > 0 then do; upper = x;   end;
    end;
    if i = 1  then LmbdCrit = x;  else LmbdCrit = LmbdCrit || x;
  end;

%*--------------------------------------------------------------*;
%* Posterior CDF of lambda.                                     *;
%*--------------------------------------------------------------*;
  do i = 1 to nlam;
    v = exp(-100<>(100><log(0.000001<>cdf('GAMMA',&lambdas.[i],a1,1/b1))));
    y = exp(-100<>(100><log(0.000001<>cdf('GAMMA',&lambdas.[i],a2,1/b2))));
    if i = 1 then CDFpsLm = v#ProbH0 + y#(1-ProbH0);
      else CDFpsLm = CDFpsLm || (v#ProbH0 + y#(1-ProbH0));
  end;
%*--------------------------------------------------------------*;
%* Get Bayes factors.                                           *;
%*--------------------------------------------------------------*;
  CDFpreLmbd = ((&alpha+1)#cdf('GAMMA',&lambdas,&nullprior.[1],1/&nullprior.[2])
        + cdf('GAMMA',&lambdas,&altprior.[1] , 1/&altprior.[2]))/(&alpha+2);

  postOdds = (1-CDFpsLm)/CDFpsLm;
  preOdds   = ((1-CDFpreLmbd) /CDFpreLmbd)`;
  do i = 1 to nlam;
    if i = 1
    then BFpost = postOdds[,i]/preOdds[i];
    else BFpost = BFpost||(postOdds[,i]/preOdds[i]);
  end;

%*-----------------------------------------------------------------------*;
%* save pargs and PstDen_p into SAS dataset "Bayes1".                    *;
%* save ProbH0, BFpost, CDFpsLm and LmbdCrit into SAS dataset "Bayes2".  *;
%*-----------------------------------------------------------------------*;
  ds1 = pargs||PstDen_p;
  varname1 = {Pargs PstDen_p};
  create Bayes1 from ds1 [colname = varname1];  append from ds1;

  ds2 = ProbH0||BFpost||CDFpsLm||(LmbdCrit)`;
  varname2 = {ProbH0}||'BFpost'+left(char(&lambdas.`))
                 ||'CDFpsLm'+left(char(&lambdas.`))||{LmbdCrit};
  create Bayes2 from ds2 [colname = varname2];  append from ds2;

%mend BayesCalcs;
%****************************************************************;
%*   MACRO NAME: GetMode.sas                                    *;
%*     FUNCTION: to calculate the mode of the posterior density *;
%*               of p, the probability that the lambda corres-  *;
%*               ponding to a drug-event  pair is generated by  *;
%*               nullprior parameter values as opposed to the   *;
%*               altprior values. It also calculates the        *;
%*               lower/upper limits of the range of values for  *;
%*               which the posterior  density of p is 'nonzero'.*;
%*         AUTHOR: Lawrence Gould                               *;
%*     PROGRAMMER: Shuping Zhang (managed by Amy Gillespie)     *;
%* DATE COMPLETED: 26FEB2008 (translate R code to SAS)          *;
%*      REVISIONS: Modified to allow for weighted sums 10/28/08 *;
%*    SAS VERSION: 9.13 or higher                               *;
%*  PLATFORM USED: Microsoft Windows XP                         *;
%*    ASSUMPTIONS: this macro will be called by %BayesCalcs.    *;
%*     INPUT DATA: refer the macro parameters below.            *;
%*   MACRO OUTPUT: mode_p, pL, pU, f1 and f2.                   *;
%*                 mode_p = mode of posterio density.           *;
%*                 pL = lower limit of range of useful p values.*;
%*                 pU = upper limit of range of useful p values.*;
%*                 f1, f2 = negative binomial density values    *;
%*                          corresponding to observed and       *;
%*                          expected drug-event report counts   *;
%*                          assuming nullprior and altprior     *;
%*                          parameter values.                   *;
%****************************************************************;
%macro getmode(
 N  = /* a vector of observed drug-event report counts.         */
,E  = /* a vector of expected drug-event report counts.         */
,wts  = /* vector of frequencies of occurrence of unique (N,E)  */
        /* pairs                                                */
,nullprior= /* parameters (a0,b0) of gamma distn for lambda     */
            /* under 'null'.                                    */
,altprior = /* parameters (a1,b1) of gamma distn for lambda     */
            /* under 'alternative'.                             */
,lims    = /* initial proposals for the limits on log(p/(1-p)). */
,alpha   = /* parameter of prior for p (beta(alpha+1,1)).       */
,cc  = /* number of log units such that if difference between   */
       /* logarithm of the posterior density of p evaluated at  */
       /* the mode and (pL,pU) is at least equal to cc, then    */
       /* the limits for the pValues range for which the        */
       /* posterior density is 'nonzero' is (pL,pU).            */
,d  = /* value of step down (for pL) or up (for pU) to determine*/
      /* when density difference of at least cc is reached.     */
);

%*--------------------------------------------------------------*;
%* set macro parameters (lims alpha cc d) default               *;
%* if user leaves them blank.                                   *;
%*--------------------------------------------------------------*;
%if "&lims" eq "" %then %let lims={-300 300};
%if "&alpha" eq "" %then %let alpha=3;
%if "&cc" eq "" %then %let cc=7;
%if "&d" eq "" %then %let d=0.01;

%*--------------------------------------------------------------*;
%* output f1 and f2 - the negative binomial density values      *;
%* corresponding to the observed and expected drug-event report *;
%* counts using nullprior / altprior.                           *;
%*--------------------------------------------------------------*;

n1=&nullprior.[1]; n2=&altprior.[1];
b1=&nullprior.[2]; b2=&altprior.[2];
q1=b1/(&E+b1);     q2=b2/(&E+b2);

c1 = lgamma(n1);  c2 = lgamma(n2);
vmN = 1:max(&N);  lgmN = lgamma(1+vmN);
gamrat1 = lgamma(n1+vmN) - lgmN - c1;
gamrat2 = lgamma(n2+vmN) - lgmN - c2;
f1 = gamrat1[&N] + n1*log(q1) + &N#log(1-q1);
f2 = gamrat2[&N] + n2*log(q2) + &N#log(1-q2);
f1 = exp(f1);  f2 = exp(f2);

%*--------------------------------------------------------------*;
%* calculate the mode of posterior density.                     *;
%*--------------------------------------------------------------*;
y=f2-f1;
lower=&lims.[1];
upper=&lims.[2];
tel=0;
do while ( round(upper-lower,.0000001)>0.0000001 & tel<1000000);
   tel=tel+1;
   p=(lower+upper)/2;
   fn=sum((&wts#y)/(f1+f2#exp(-p))) - &alpha;
   if fn=0 then do; tel=10000; end; else
   if fn<0 then do; lower=p;   end; else
   if fn>0 then do; upper=p;   end;
end;
epp = exp(-p);  mode_p = 1/(1+exp(-p));

%*--------------------------------------------------------------*;
%* calculate the range enclosing nonzero values.                *;
%*--------------------------------------------------------------*;
sw = sum(&wts);
fmode = &wts`*log(f1 + epp#f2) - (&alpha + sw)*log(1+epp);

ppL = p;  fL = fmode;
do while  ((fmode - fL) < &cc);
  ppL = ppL - &d;  epp = exp(-ppL);
  fL = &wts`*log(f1 + epp#f2) - (&alpha + sw)*log(1+epp);
end;
pL = 1/(1 + epp);

ppU = p;  fU = fmode;
do while  ((fmode - fU) < &cc);
  ppU = ppU + &d;  epp = exp(-ppU);
  fU = &wts`*log(f1 + epp#f2) - (&alpha + sw)*log(1+epp);
end;
pU = 1/(1 + epp);

/*   print mode_p fmode pL fL pU fU;                 */;
%mend getmode;

proc iml;

USE &datain..&filein.U2;
READ all var{N E wt};
CLOSE &datain..&filein.U2;

%let lbda_0=2; /* DEFINE THIS VALUES, DEFAULT = 2*/

%BayesCalcs( N = N
, E =  E
, wts = wt
, nullprior = {2 1.65} /* Gould's 2007 article, page 159 */
, altprior = {2 0.2}, numpargs =,lambdas ={&lbda_0},alpha  =, targ_p =);
quit;

data &datain..&filein.U2 (rename=(BFPost&lbda_0=score));
merge &datain..&filein.U2 Bayes2;
keep N E BFPost&lbda_0;
run;


proc sort data=&datain..&filein.U1; by N E; run;
proc sort data=&datain..&filein.U2; by N E; run;

data &dataout..&filein.m5 (keep = Drug_concept_id Condition_concept_id score);
merge &datain..&filein.U1 &datain..&filein.U2;
by N E;
run;

data &dataout..&filein.m5;
set &dataout..&filein.m5;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m5;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m5;
        merge &dataout..&filein.m5 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;



%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m5.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m5   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_5;

%macro dpm_6(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  URN macro,                                                     *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/
%let st=%eval(&sex_s+&age_s+&year_s);

%if &st = 0 %then

%do;
proc sort data=&datain..&filein (where =((Drug_concept_id NE -999) AND (Condition_concept_id NE -999)))
out=&datain..&filein._6 (keep=W00 W01 W10 W11) NODUPKEY;
by W00 W01 W10 W11;
run;

data &datain..&filein._6; set &datain..&filein._6;
tn=_n_; run;

data &datain..&filein._61;
set &datain..&filein._6;
array XX[4] W00 W01 W10 W11;
do i=1 to 4;
count=XX[i];
if i=1 then do; grp ='1'; resp='YES'; end;
  else if i=2 then do; grp ='1'; resp='_NO'; end;
    else; if i=3 then do; grp ='2'; resp='YES'; end;
          else; if i=4 then do; grp ='2'; resp='_NO'; end;
output;
end;
drop i W00 W01 W10 W11;
run;

proc freq data=&datain..&filein._61 NOPRINT;
tables grp*resp / fisher NOWARN;
weight count;
by tn;
output out=&datain..&filein._62 fisher exact;
run;

data &datain..&filein._63;
merge &datain..&filein._6 &datain..&filein._62;
by tn;
keep W00 W01 W10 W11 XP2_FISH;
run;

proc sort data=&datain..&filein; by W00 W01 W10 W11; run;
proc sort data=&datain..&filein._63; by W00 W01 W10 W11; run;

data &dataout..&filein.m6;
merge &datain..&filein (where =((Drug_concept_id NE -999) AND (Condition_concept_id NE -999)))
&datain..&filein._63;
by W00 W01 W10 W11;
if XP2_FISH NE 0 then score=1/XP2_FISH;
   else score =1.E150;
drop XP2_FISH stratum W00 W01 W10 W11 N;
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m6;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m6;
        merge &dataout..&filein.m6 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m6.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m6   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%end;
%else %put This version of DPM_6 does not allow stratification;


%mend dpm_6;



%macro dpm_7(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  SIGNED CHI-SQUARE macro,                                        *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/

proc sql;
create table &dataout..&filein.m7 as
select T1.Drug_concept_id, T1.Condition_concept_id,
SIGN(sum(W00)- sum((W00+W01)*(W00+W10)/N))*((sum(W00)- sum((W00+W01)*(W00+W10)/N))**2)/sum((W00+W01)*(W10+W11)*(W00+W10)*(W01+W11)/(N*N*(N-1))) as score
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;

data &dataout..&filein.m7;
set &dataout..&filein.m7;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m7;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m7;
        merge &dataout..&filein.m7 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m7.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m7   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_7;


%macro dpm_8(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  PRR05 macro,                                                    *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/

proc sql;
create table &datain..&filein.m8_1 as
select T1.Drug_concept_id, T1.Condition_concept_id, sum(W00*(W10 + W11)/N)/sum(W10*(W00 + W01)/N) as PRR,
sqrt(sum(((W00+W01)*(W10+W11)*(W00+W10)-W00*W10*N)/(N*N))/(sum(W00*(W10+W11)/N)*(sum(W10*(W00+W01)/N)))) as sigma
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;

data &dataout..&filein.m8;
set &datain..&filein.m8_1;
score=PRR*EXP(-1.645*sigma);
drop PRR sigma;
run;

data &dataout..&filein.m8;
set &dataout..&filein.m8;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m8;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m8;
        merge &dataout..&filein.m8 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m8.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m8   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_8;


%macro dpm_9(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  ROR05 macro,                                                    *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/

proc sql;
create table &datain..&filein.m9_1 as
select T1.Drug_concept_id, T1.Condition_concept_id, sum(W00*W11/N)/sum(W10*W01/N) as ROR,
sum((W00+W11)*(W00*W11)/(N*N))/(2*((sum(W00*W11/N))**2)) as sigma21,
sum(((W10+W11)*(W01*W10)+(W01+W10)*(W00*W11))/(N*N))/(2*sum(W00*W11/N)*sum(W01*W10/N)) as sigma22,
sum((W01+W10)*(W01*W10)/(N*N))/(2*((sum(W01*W10/N))**2)) as sigma23
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;


data &dataout..&filein.m9;
set &datain..&filein.m9_1;
score=ROR*EXP(-1.645*sqrt(sigma21+sigma22+sigma23));
drop ROR sigma21 sigma22 sigma23;
run;

data &dataout..&filein.m9;
set &dataout..&filein.m9;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&dataout..&filein.m9;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &dataout..&filein.m9;
        merge &dataout..&filein.m9 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m9.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m9   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_9;

%macro dpm_10(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  BCPNN05 macro,                                                  *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/

proc sql;
create table &datain..&filein.m10 as
select T1.Drug_concept_id, T1.Condition_concept_id,
sum(W00)+0.5 as N_ij, (sum((W00+W01)*(W00+W10)/N) + 0.5) as E_ij
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;


/* create output pairs if defined */

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&datain..&filein.m10;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &datain..&filein.m10;
        merge &datain..&filein.m10 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;


proc iml;

USE &datain..&filein.m10;
READ all var{N_ij E_ij};
CLOSE &datain..&filein.m10;

start pi_b(lambda, alpha, beta) global(N_);
        A=0;
        B=lambda;
        if lambda >10 then N_=1000;
    h=(B-A)/N_;
        x1=A+((1:N_)-0.5)*h;
        x2=A+(1:(N_-1))*h;
        f_a=PDF('GAMMA', A, alpha, 1/beta);
        f_b=PDF('GAMMA', B, alpha, 1/beta);
        f_05h=PDF('GAMMA', x1, alpha, 1/beta);
        f_h=PDF('GAMMA', x2, alpha, 1/beta);
        y=h*(f_a+4*sum(f_05h)+2*sum(f_h)+f_b)/6;
        y=y-0.05;
        return(y);
finish pi_b;

M=nrow(N_ij); lambda05=J(M,1,0);

/* BISECTION */
do i=1 to M;
    N_=50;
        alpha=N_ij[i];  beta=E_ij[i];
        moda=(alpha-1)/beta;
        lambda0=moda/16;
        y = pi_b(lambda0, alpha, beta);
        xl=0; xr=lambda0;

/* Find starting points */
        
                do while(y <0);
                lambda0=lambda0*2;
                y=pi_b(lambda0, alpha, beta);
                xl=lambda0/2; xr=lambda0;
                end;
/*precision */
        eps=1.e-5;
*Start loop;
        iter=0;
                Do while(abs(xr - xl) > 2*eps);
                        yl=pi_b(xl,alpha, beta);
                        midx = (xr + xl) / 2;
                        ym=pi_b(midx,alpha, beta);
                        If (yl * ym > 0) then
                        do;     xl = midx; end;
                        else    xr = midx;
                iter=iter+1;
                        if iter >100000 then do; print "too many iterations in dpm_10"; abort; end;
                End;
         x=(xl + xr) / 2;
        lambda05[i]=log2(x);
end;

create &datain..lambda05 from lambda05[colname='score']; append from lambda05; close &datain..lambda05;

data &dataout..&filein.m10;
merge &datain..&filein.m10 &datain..lambda05;
drop N_ij E_ij; run;

quit;

data &dataout..&filein.m10;
set &dataout..&filein.m10;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;


data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m10.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m10   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_10;




%macro dpm_11(datain, dataout, filein, dr_ind, cnd_ind);
/*******************************************************************
*  EB05 macro,                                                     *
*  datain - library that contains input data;                      *
*  dataout - library that stores output file;                      *
*  filein  - input file name;                                      *
*  dr_ind - indicator of drugs__of_interest.txt file               *
*  cnd_ind - indicator of conditions_of_interest file              *
*******************************************************************/

proc sql;
create table &datain..&filein.EBGM1 as
select T1.Drug_concept_id, T1.Condition_concept_id, sum(T1.W00) as N_ij,
sum((W00+ W01)*(W00+W10)/N) as E_ij
from  &datain..&filein T1
group by T1.Drug_concept_id, T1.Condition_concept_id;
quit;



/* create output pairs if defined */

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond ^= 0 %then
%do;
        proc sort data=&datain..&filein.EBGM1;
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        run;

        data &datain..&filein.E12 (rename=(N_ij=N_ij_2 E_ij=E_ij_2));
        merge &datain..&filein.EBGM1 (IN=SET1) &datain..Outputpairs (IN=SET2);
        by
        %if &dr_ind=1 %then %do; Drug_concept_id %end;
        %if &cnd_ind=1 %then %do; Condition_concept_id %end;
        ;
        if SET1 AND SET2;
        run;
%end;

%else  %do; data &dataout..&filein.E12 (rename=(N_ij=N_ij_2 E_ij=E_ij_2)); set &dataout..&filein.EBGM1; run; %end;




proc iml;
/*******************************************************************
*   Optimization procedure for a mixture of                        *
*   negative binomial densities                                    *
*******************************************************************/


USE &datain..&filein.EBGM1;  READ all var{N_ij E_ij}; CLOSE &datain..&filein.EBGM1;
USE &datain..&filein.E12;  READ all var{N_ij_2 E_ij_2}; CLOSE &datain..&filein.E12;


MAX_N=max(N_ij); m=nrow(N_ij);


/*******************************************************************
*   OBJECTIVE FUNCTION                                             *
*******************************************************************/
start NEGBIN_I(x) global(N_ij, E_ij, MAX_N, m);

x1=x[1]; x2=x[2]; x3=x[3]; x4=x[4]; x5=x[5];

GNB1=J(MAX_N,1,0); GNB3=J(MAX_N,1,0); GS1=J(MAX_N,1,0);
GS3=J(MAX_N,1,0); G2S1=J(MAX_N,1,0); G2S3=J(MAX_N,1,0);

pro1=1; pro3=1;
sum_1=0; sum_3=0; cum_1=0; cum_3=0;

do i=1 to MAX_N by 1;
        pro1=pro1*((x1-1)/i+1);
        pro3=pro3*((x3-1)/i+1);
        GNB1[i]=pro1;
        GNB3[i]=pro3;
        sum_1=sum_1+1/(i+x1-1);
        sum_3=sum_3+1/(i+x3-1);
        GS1[i]=sum_1;
        GS3[i]=sum_3;
        cum_1=cum_1+1/((i+x1-1)**2);
        cum_3=cum_3+1/((i+x3-1)**2);
        G2S1[i]=cum_1;
        G2S3[i]=cum_3;
end;

p1=E_ij/(E_ij+x2); p2=E_ij/(E_ij+x4);
q1=1-p1; q2=1-p2;
GNBA1=GNB1[N_ij];
GNBA3=GNB3[N_ij];
fa1=GNBA1#(p1##N_ij)#(q1##x1);
fa2=GNBA3#(p2##N_ij)#(q2##x3);
fa=x5#fa1+(1-x5)#fa2;

mc=1.e-150; c=(fa >mc); c1=1-c;
fa=fa#c + c1#mc; fa=log(fa);
y1=sum(fa);
return(y1);
finish NEGBIN_I;


/*******************************************************************
*   DERIVATIVE/GRADIENT                                            *
*******************************************************************/
start G_NEGBIN_I(x) global(N_ij, E_ij, MAX_N, m);
print x;
g=j(1,5,0);
x1=x[1]; x2=x[2]; x3=x[3]; x4=x[4]; x5=x[5];

GNB1=J(MAX_N,1,0); GNB3=J(MAX_N,1,0); GS1=J(MAX_N,1,0);
GS3=J(MAX_N,1,0); G2S1=J(MAX_N,1,0); G2S3=J(MAX_N,1,0);

pro1=1; pro3=1;
sum_1=0; sum_3=0; cum_1=0; cum_3=0;

do i=1 to MAX_N by 1;
        pro1=pro1*((x1-1)/i+1);
        pro3=pro3*((x3-1)/i+1);
        GNB1[i]=pro1;
        GNB3[i]=pro3;
        sum_1=sum_1+1/(i+x1-1);
        sum_3=sum_3+1/(i+x3-1);
        GS1[i]=sum_1;
        GS3[i]=sum_3;
        cum_1=cum_1+1/((i+x1-1)**2);
        cum_3=cum_3+1/((i+x3-1)**2);
        G2S1[i]=cum_1;
        G2S3[i]=cum_3;
end;

p1=E_ij/(E_ij+x2); p2=E_ij/(E_ij+x4);
q1=1-p1; q2=1-p2;
mc=1.e-150;
cq1=(q1 > 0); cq11=1-cq1;
q1=q1#cq1 + cq11#mc;
cq2=(q2 > 0); cq21=1-cq2;
q2=q2#cq2 + cq21#mc;

GNBA1=GNB1[N_ij]; GNBA3=GNB3[N_ij];
fa1=GNBA1#(p1##N_ij)#(q1##x1);
fa2=GNBA3#(p2##N_ij)#(q2##x3);
fa=x5#fa1+(1-x5)#fa2;

mc=1.e-150; c=(fa >mc); c1=1-c;
fa=fa#c + c1#mc;

r2a=GS1[N_ij]+log(q1); f_by_alpha12=fa1#r2a;
sum1=sum(x5#f_by_alpha12/fa); r1a=fa1#p1#q1;
r2a2=-1#N_ij/(E_ij#q1)+x1#E_ij/(x2#x2#p1);
f_by_beta12=r1a#r2a2;
sum2=sum(x5#f_by_beta12/fa);

r2a3=GS3[N_ij]+log(q2); f_by_alpha34=fa2#r2a3;
sum3=sum((1-x5)#f_by_alpha34/fa);

r1a4=fa2#p2#q2; r2a4=-1#N_ij/(E_ij#q2)+x3#E_ij/(x4#x4#p2);
f_by_beta34=r1a4#r2a4;
sum4=sum((1-x5)#f_by_beta34/fa);
sum5=sum((fa1-fa2)/fa);

g[1]=sum1; g[2]=sum2; g[3]=sum3; g[4]=sum4; g[5]=sum5;
return(g);
finish G_NEGBIN_I;


/*******************************************************************
*   HESSIAN                                                        *
*******************************************************************/
start H_NEGBIN_I(x) global(N_ij, E_ij, MAX_N, m);

hes=j(5,5,0);
x1=x[1]; x2=x[2]; x3=x[3]; x4=x[4]; x5=x[5];

GNB1=J(MAX_N,1,0); GNB3=J(MAX_N,1,0); GS1=J(MAX_N,1,0);
GS3=J(MAX_N,1,0); G2S1=J(MAX_N,1,0); G2S3=J(MAX_N,1,0);

pro1=1; pro3=1;
sum_1=0; sum_3=0; cum_1=0; cum_3=0;

do i=1 to MAX_N by 1;
        pro1=pro1*((x1-1)/i+1);
        pro3=pro3*((x3-1)/i+1);
        GNB1[i]=pro1;
        GNB3[i]=pro3;
        sum_1=sum_1+1/(i+x1-1);
        sum_3=sum_3+1/(i+x3-1);
        GS1[i]=sum_1;
        GS3[i]=sum_3;
        cum_1=cum_1+1/((i+x1-1)**2);
        cum_3=cum_3+1/((i+x3-1)**2);
        G2S1[i]=cum_1;
        G2S3[i]=cum_3;
end;
onex5=1-x5;

p1=E_ij/(E_ij+x2); p2=E_ij/(E_ij+x4);
q1=1-p1; q2=1-p2;
mc=1.e-150;
cq1=(q1 > 0); cq11=1-cq1;
q1=q1#cq1 + cq11#mc;
cq2=(q2 > 0); cq21=1-cq2;
q2=q2#cq2 + cq21#mc;

GNBA1=GNB1[N_ij]; GNBA3=GNB3[N_ij];
GS1A=GS1[N_ij]; G2S1A=G2S1[N_ij];
GS3A=GS3[N_ij]; G2S3A=G2S3[N_ij];

fa1=GNBA1#(p1##N_ij)#(q1##x1);
fa2=GNBA3#(p2##N_ij)#(q2##x3);
fa=x5#fa1+(1-x5)#fa2;

mc=1.e-150; c=(fa >mc); c1=1-c;
fa=fa#c + c1#mc;
delf12 = fa1 - fa2;
den=fa##2;

r2_aa1=GS1A + log(q1); fbyaa12=fa1#(r2_aa1##2- G2S1A);
r2_aa2=GS3A+log(q2); fbyaa34=fa2#(r2_aa2##2-G2S3A);

r1_ab1=fa1#p1; r3_ab1=(x1/x2-N_ij/E_ij);
fbyab12=r1_ab1#(1/x2+r2_aa1#r3_ab1);

r1_ab2=fa2#p2; r3_ab2=(x3/x4-N_ij/E_ij);
fbyab34=r1_ab2#(1/x4+r2_aa2#r3_ab2);

fbyalpha12=fa1#r2_aa1; fbyalpha34=fa2#r2_aa2;

r1_beta1=r1_ab1#q1; r2_beta1=-1#N_ij/(E_ij#q1)+x1#E_ij/(x2#x2#p1);
fbybeta12=r1_beta1#r2_beta1;

r1_beta2=r1_ab2#q2; r2_beta2=-1#N_ij/(E_ij#q2)+x3#E_ij/(x4#x4#p2);
fbybeta34=r1_beta2#r2_beta2;

r2=1/(x2#(E_ij+x2)); r3=(x1#E_ij - x2#N_ij)/(E_ij+x2);
r4=(x1#E_ij)/x2 - N_ij; r5=N_ij#(x2##2)-2#x1#x2#E_ij - x1#(E_ij##2);
fbybb12=fa1#r2#(r3#r4+r5#r2);

r2=1/(x4#(E_ij+x4)); r3=(x3#E_ij-x4*N_ij)/(E_ij+x4);
r4=(x3#E_ij)/x4-N_ij;
r5=N_ij#(x4##2)-2#x3#x4#E_ij-x3#(E_ij##2);
fbybb34=fa2#r2#(r3#r4+r5#r2);

sum11 = sum((x5#fbyaa12#fa-(x5#fbyalpha12)##2)/den);
sum12 = sum(x5#(fbyab12#fa - fbyalpha12#x5#fbybeta12)/den);
sum13 = -1*sum(x5#fbyalpha12#onex5#fbyalpha34/den);
sum14 = -1*sum(x5#fbyalpha12#onex5#fbybeta34/den);
sum15 = sum((fbyalpha12#fa - x5#fbyalpha12#delf12)/den);
sum22 = sum((x5#fbybb12#fa-(x5#fbybeta12)##2)/den);
sum23 = -1*sum(x5#fbybeta12#onex5#fbyalpha34/den);
sum24 = -1*sum(x5#fbybeta12#onex5#fbybeta34/den);
sum25 = sum((fbybeta12#fa-x5#fbybeta12#delf12)/den);
sum33 = sum((onex5#fbyaa34#fa - (onex5#fbyalpha34)##2)/den);
sum34 = sum(onex5#(fbyab34#fa - fbyalpha34#onex5#fbybeta34)/den);
sum35 = -1*sum((fbyalpha34#fa + onex5#fbyalpha34#delf12)/den);
sum44 = sum((onex5#fbybb34#fa - (onex5#fbybeta34)##2)/den);
sum45 = -1*sum((fbybeta34#fa + onex5#fbybeta34#delf12)/den);
sum55 = -1*sum((delf12##2)/den);

hes[1,1]=sum11; hes[1,2]=sum12; hes[1,3]=sum13; hes[1,4]=sum14; hes[1,5]=sum15;
hes[2,1]=sum12; hes[2,2]=sum22; hes[2,3]=sum23; hes[2,4]=sum24; hes[2,5]=sum25;
hes[3,1]=sum13; hes[3,2]=sum23; hes[3,3]=sum33; hes[3,4]=sum34; hes[3,5]=sum35;
hes[4,1]=sum14; hes[4,2]=sum24; hes[4,3]=sum34; hes[4,4]=sum44; hes[4,5]=sum45;
hes[5,1]=sum15; hes[5,2]=sum25; hes[5,3]=sum35; hes[5,4]=sum45; hes[5,5]=sum55;
return(hes);
finish H_NEGBIN_I;


/*******************************************************************
*   x0 - initial value for optimization procedure                  *
*******************************************************************/
x0={0.2 0.1 2 4 0.3333};
con={1.e-6 1.e-6 1.e-6 1.e-6 0,
. . . . 1};
optn={1 2};

par={. 0.5 . . . . . . . .};
call nlpnra(rc,xres,"NEGBIN_I",x0,optn,con,,par,,"G_NEGBIN_I","H_NEGBIN_I");
print "PRINT RC=" rc;

a1=xres[1]; b1=xres[2]; a2=xres[3]; b2=xres[4]; P=xres[5];
GNB1=J(MAX_N,1,0); GNB3=J(MAX_N,1,0);

pro1=1; pro3=1;

do i=1 to MAX_N by 1;
        pro1=pro1*((a1-1)/i+1);
        pro3=pro3*((a2-1)/i+1);
        GNB1[i]=pro1;
        GNB3[i]=pro3;
end;

p1=E_ij/(E_ij+b1); p2=E_ij/(E_ij+b2);
q1=1-p1; q2=1-p2;
GNBA1=GNB1[N_ij];
GNBA3=GNB3[N_ij];
fa1=GNBA1#(p1##N_ij)#(q1##a1);
fa2=GNBA3#(p2##N_ij)#(q2##a2);
fa=P#fa1+(1-P)#fa2;

mc=1.e-150; c=(fa >mc); c1=1-c;
fa=fa#c + c1#mc;
Qn=P#fa1/fa;

n1=N_ij_2+a1; n2=N_ij_2+a2;
e1=E_ij_2+b1; e2=E_ij_2+b2;
m=nrow(N_ij_2);
/*******************************************************************
*   EB05                                                           *
*******************************************************************/
start pi(lambda, alpha1, beta1, alpha2, beta2, P)  global(N_);

        A=0; B=lambda;
        if lambda >10 then N_=1000;
        h=(B-A)/N_;
        x1=A+((1:N_)-0.5)*h;    x2=A+(1:(N_-1))*h;

        f_a=P*PDF('GAMMA', A, alpha1, 1/beta1)+ (1-P)*PDF('GAMMA', A, alpha2, 1/beta2);
        f_b=P*PDF('GAMMA', B, alpha1, 1/beta1)+ (1-P)*PDF('GAMMA', B, alpha2, 1/beta2);
        f_05h=P*PDF('GAMMA', x1, alpha1, 1/beta1)+ (1-P)*PDF('GAMMA', x1, alpha2, 1/beta2);
        f_h=P*PDF('GAMMA', x2, alpha1, 1/beta1)+ (1-P)*PDF('GAMMA', x2, alpha2, 1/beta2);

        y=h*(f_a+4*sum(f_05h)+2*sum(f_h)+f_b)/6; y=y-0.05;
        return(y);
finish pi;


lambda05=J(m,1,0);

do i=1 to m;
   N_=50;
        alpha1=n1[i]; beta1=e1[i];
        alpha2=n2[i]; beta2=e2[i];
        P=Qn[i];
    moda1=(alpha1-1)/beta1; moda2=(alpha2-1)/beta2;
        modas=min(moda1,moda2);
        lambda0=modas/16;
        y = pi(lambda0, alpha1, beta1, alpha2, beta2, P);
        xl=0; xr=lambda0;
    
                do while(y <0);
                lambda0=lambda0*2;
                y=pi(lambda0, alpha1, beta1, alpha2, beta2, P);
                xl=lambda0/2; xr=lambda0;
                                end;

        eps=1.e-5;

        iter=0;
        Do while(abs(xr - xl) > 2*eps);
        yl=pi(xl, alpha1, beta1, alpha2, beta2, P);

        midx = (xr + xl) / 2;
        ym=pi(midx, alpha1, beta1, alpha2, beta2, P);

        If (yl * ym > 0) Then
    xl = midx;
        Else
    xr = midx;
    iter=iter+1;
        if iter >100000 then do; print "too many iterations in dpm_11"; abort; end;
        End;

        x=(xl + xr) / 2;

        lambda05[i]=x;
end;

create &datain..lambda05 from lambda05[colname='score']; append from lambda05; close &datain..lambda05;

data &dataout..&filein.m11;
merge &datain..&filein.E12 &datain..lambda05;
drop N_ij_2 E_ij_2; run;

quit;

data &dataout..&filein.m11;
set &dataout..&filein.m11;
if score EQ . then delete;
where (Drug_concept_id NE -999) AND (Condition_concept_id NE -999);
run;


%if &s_window<0 %then %do; %let SW=%eval(-1*&s_window); %let s_windo=n&SW; %end; %else %let s_windo=&s_window;
%let fileout_n=DP&pvn._&dbn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;



data _null_;
    %let _EFIERR_ = 0; 
    %let _EFIREC_ = 0; 
    file "&TEXT_OUT_DIR./&fileout_n.m11.txt" delimiter=',' DSD DROPOVER lrecl=32767;
    if _n_ = 1 then    
     do;
       put
          "Drug_concept_id"
       ','
          "Condition_concept_id"
       ','
          "score"
       ;
     end;
   set  &dataout..&filein.m11   end=EFIEOD;
       format Drug_concept_id best12. ;
       format Condition_concept_id best12. ;
       format score best12. ;
     do;
       EFIOUT + 1;
       put Drug_concept_id @;
       put Condition_concept_id @;
       put score ;
       ;
     end;
    if _ERROR_ then call symputx('_EFIERR_',1);  
    if EFIEOD then call symputx('_EFIREC_',EFIOUT);
    run;

%mend dpm_11;


%macro DP(datain, datawrk, dataout);
/*******************************************************************
* MACRO DP: Calculates DP metrics and stores them in the output    *
* library, calls counting scenarios and metrics;                   *
* INPUT PARAMETERS:                                                *
* datawrk - library that stores working files and preprocessed     *
* data in the cdm format                                           *
* dataout - library for the output file(s)                         *
*******************************************************************/

%let pvn=31; /* DP program version 3.1 */


/* Read txt file. Macro uses code from the Carpenter's guide to the SAS macro */
%let filrf=myfile;
%let rc=%sysfunc(filename(filrf, &DP_parameters));
%let fid=%sysfunc(fopen(&filrf));
%let rc=%sysfunc(fsep(&fid,%str(:,)));
%let i=1;
%if &fid > 0 %then
   %do %while(%sysfunc(fread(&fid)) = 0);
       %let rc=%sysfunc(fget(&fid,c,200));
       %let b&i = &c;
       %let i=%eval(&i+1);
    %end;
      %put &b1 &b2 &b2 &b3 &b4 &b5 &b6 &b7 &b8 &b9 &b10 &b11 &b12;
   %let rc=%sysfunc(fclose(&fid));
%let rc=%sysfunc(filename(filrf));



%macro nw(string);

%local count word;
%global nw word1;

%let count=1;

%let word=%qscan(&string,&count,%str(:;, ));
%let word1=&word;

%do %while(&word ne);
   %let count=%sysevalf(&count + 1);
   %let word=%qscan(&string,&count,%str(:;, ));
   %global word&count;
   %let word&count=&word;
%end;

%let nw=%eval(&count-1);
%mend nw;

%nw(&b1); %let TOTCN=%eval(&nw-1); %do j=1 %to &TOTCN; %let k=%eval(&j+1); %let CON&j  = %eval(&&word&k); %put type &&CON&j; %end;
%nw(&b2); %let TOTCS=%eval(&nw-1); %do j=1 %to &TOTCS; %let k=%eval(&j+1); %let CVRS&j = %eval(&&word&k); %put count &&CVRS&j; %end;
%nw(&b3); %let TOTMD=%eval(&nw-1); %do j=1 %to &TOTMD; %let k=%eval(&j+1); %let MTHD&j = %eval(&&word&k); %put metric &&MTHD&j; %end;
%nw(&b4); %let age_s=%eval(&word2); %put age &age_s;
%nw(&b5); %let sex_s=%eval(&word2); %put sex  &sex_s;
%nw(&b6); %let year_s=%eval(&word2); %put year &year_s;
%nw(&b7); %let s_window=%eval(&word2); %put S_WINDOW &s_window;
%nw(&b8); %let p_window_drugs=%eval(&word2); %put DRUG_WINDOW &p_window_drugs;
%nw(&b9); %let p_window_cond=%eval(&word2); %put COND_WINDOW &p_window_cond;
%nw(&b10); %let Drug_era_table=%scan(&word2,1); %put Drug_era_table &Drug_era_table;
%nw(&b11); %let Condition_era_table=%scan(&word2,1); %put Condition_era_table &Condition_era_table;
%nw(&b12); %let dbn_f=%scan(&word2,1); %put DBN from file &dbn_f;

%if &dbn_f=NULL %then %do; %let dbn=&dbn_; %end; 
                                                                                                %else %do; %let dbn=&dbn_f; %end;

%if &p_window_drugs=0 %then %do; %let PWD='6'; %put PWD &PWD; %end;
  %else %if &p_window_drugs=30 %then %do; %let PWD='7'; %end;
              %else %put ' WRONG SIZE OF THE DRUG PERSISTENCE WINDOW';


/*******************************************************************
*   preprocess input data, Drug_era                                *
*******************************************************************/
proc sort data=&DATAIN..&Drug_era_table (where=(Drug_Exposure_Type = &PWD) keep=Person_id Drug_era_start_date Drug_era_end_date
Drug_concept_id Drug_Exposure_Type) out=&DATAWRK..Drug_era (drop=Drug_Exposure_Type)  NODUPKEY THREADS;
by Person_id Drug_concept_id Drug_era_start_date Drug_era_end_date;
run;


%if &s_window <0 %then
%do;
    %put 'window negative';  
        data &DATAWRK..drugs;
        set &DATAWRK..drug_era;
        drug_era_end_date=drug_era_start_date - &s_window;
        keep person_id drug_concept_id drug_era_start_date drug_era_end_date;
        run;
%end;

%if &s_window > 0 %then
%do;
    %put 'window positive';  
        data &DATAWRK..drugs;
        set &DATAWRK..drug_era;
        drug_era_end_date=drug_era_end_date + &s_window;
        keep person_id drug_concept_id drug_era_start_date drug_era_end_date;
        run;
%end;

%if &s_window ^= 0 %then
%do;

proc sort data=&DATAWRK..drugs;
by person_id drug_concept_id drug_era_start_date;
run;


data &DATAWRK..drugs2;

retain end_date  -10000000000000000000000000000000;
retain start_date 10000000000000000000000000000000;
retain did 0;
retain person 0;

set &DATAWRK..drugs;

if person NE person_id then person=0;
if did NE drug_concept_id then did =0;

if (did EQ 0) OR (person=0) then do; did=drug_concept_id; person=person_id; 
end_date=-10000000000000000000000000000000;
start_date=10000000000000000000000000000000; end;

if end_date < drug_era_start_date then do; end_date=drug_era_end_date; start_date=drug_era_start_date; end;
if end_date >=start_date then do; drug_era_start_date=start_date; end_date=drug_era_end_date; end;
length=drug_era_end_date-drug_era_start_date;

run; 

proc sort data=&DATAWRK..drugs2;
by person_id drug_concept_id DESCENDING length drug_era_start_date;
run;


proc sort data=&DATAWRK..drugs2;
by person_id drug_concept_id drug_era_start_date;
run;


proc sort data=&DATAWRK..drugs2 out=&DATAWRK..drug_era (keep= person_id drug_concept_id drug_era_start_date drug_era_end_date) NODUPKEY;
by person_id drug_concept_id drug_era_start_date;
run;
%end;

/*******************************************************************
*   preprocess input data, Condition_era                           *
*******************************************************************/
%if &p_window_cond=0 %then %do; %let PWC='64'; %end;
  %else %if &p_window_cond=30 %then %do; %let PWC='65'; %end;
              %else %put ' WRONG SIZE OF THE CONDITION PERSISTENCE WINDOW';
proc sort data=&DATAIN..&Condition_era_table (where=(Condition_occurrence_type = &PWC) keep = Person_id Condition_era_start_date
Condition_concept_id Condition_occurrence_type) out=&DATAWRK..Condition_era (drop=Condition_occurrence_type) NODUPKEY THREADS;
by Person_id Condition_concept_id Condition_era_start_date;
run;


/*******************************************************************
*   preprocess input data: Person data, Observation_period         *
*******************************************************************/
proc sort data=&DATAIN..Person out=&DATAWRK..Person (keep=Person_id Year_of_birth Gender_concept_id) THREADS;
by Person_id;
run;

data &DATAWRK..First_record;
set &DATAIN..Observation_period;
year=year(Observation_period_start_date);
keep Person_id year;
run;

proc sort data=&DATAWRK..First_record NODUPKEY THREADS;
by Person_id;
run;



/*******************************************************************
*  Output pairs                                                    *
*******************************************************************/

%if %sysfunc(fileexist(&drugs_of_interest)) %then
%do; %put drugs_of_interest.txt file exists;
        data &DATAWRK..drugs_of_interest;
        infile "&drugs_of_interest" firstobs=2;
        input Drug_concept_id;
        run;

        %let dr_ind=1;
%end;
%else %do; %put drugs_of_interest.txt does not exist; %let dr_ind=0; %end;

%if %sysfunc(fileexist(&conditions_of_interest)) %then
%do; %put conditions_of_interest.txt file exists;
        data &DATAWRK..conditions_of_interest;
        infile "&conditions_of_interest" firstobs=2;
        input Condition_concept_id;
        run;

        %let cnd_ind=1;
%end;
%else %do; %put conditions_of_interest.txt does not exist; %let cnd_ind=0; %end;

%let both_cond=%eval(&dr_ind + &cnd_ind);
%if &both_cond=2 %then %do;
        proc sql;
        create table &DATAWRK..Outputpairs as
        select T1.Drug_concept_id, T2.Condition_concept_id from
        &DATAWRK..drugs_of_interest T1, &DATAWRK..conditions_of_interest T2;
        quit;

        proc sort data=&DATAWRK..Outputpairs THREADS;
        by Drug_concept_id Condition_concept_id;
        run;
%end;
%else
%do;
        %if &dr_ind=1 %then
                        %do;    data &DATAWRK..Outputpairs; set &DATAWRK..drugs_of_interest; run;
                                        proc sort data=&DATAWRK..Outputpairs;
                                        by Drug_concept_id;
                                        run;
                        %end;
        %if &cnd_ind=1 %then
                        %do;    data &DATAWRK..Outputpairs; set &DATAWRK..conditions_of_interest; run;
                                        proc sort data=&DATAWRK..Outputpairs;
                                        by Condition_concept_id;
                                        run;
                        %end;
%end;

/*******************************************************************
*   calculate all specified DP metrics for every combination       *
*   of the condition type and counting scenario                    *
*******************************************************************/

%do i=1 %to &TOTCN;

 %let CN=&&CON&i;


 %do m=1 %to &TOTCS;
    %let num=&&CVRS&m;  /* technical macro, number of the current counting scenario */
        %if &s_window=-30 %then %let s_windo=n30; %else %let s_windo=&s_window;
                /*DP&pvn.&CN.&num.s&s_windo.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s*/
    %let Cnm=DP&pvn.&CN.&num.d&p_window_drugs.c&p_window_cond.st&age_s.&sex_s.&year_s;
        /*&Cnm = name of the input file for a DP metrics program */
                %let Csname=c_s&num; /* name of a counting macro */
   %&Csname(&datawrk, &datawrk, &CN, &p_window_drugs, &p_window_cond, &sex_s, &age_s, &year_s, &dbn, &pvn);
            %do j=1 %to &TOTMD;
               %let mnum=&&MTHD&j;
                   %dpm_&mnum(&datawrk, &dataout, &Cnm, &dr_ind, &cnd_ind);
            %end;
   %end;
%end;

%mend DP;

/*******************************************************************
*   Call DP macro to calculate DP metrics                          *
*******************************************************************/
%DP(&datain,&datawrk,&dataout);

